=begin

= fileutils.rb

  Copyright (c) 2000,2001 Minero Aoki <aamine@loveruby.net>

  This program is free software.
  You can distribute/modify this program under the terms of
  the Ruby License.

== module FileUtils

The module which implements basic file operations.
You can also use FileUtils::Normal as alias.

=== Module Functions

--- FileUtils#cd( dir )
--- FileUtils#cd( dir ) {|dir| .... }
  changes current directory to DIR.

  If this method is called with block, moves to the old current
  directory after the block execution finished.

--- FileUtils#pwd
--- FileUtils#getwd
  returns name of the current dirctory.
  This method is same to Dir.pwd.

--- FileUtils#newest?( newer, older1, older2, ... )
--- FileUtils#uptodate?( newer, older1, older2, ... )
  true if NEWER is newer than all OLDERs.
  Non-exist files are older than any file.

    FileUtils.newest? 'hello.o', 'hello.c', 'hello.h' or system 'make'

--- FileUtils#oldest?( older, newer1, newer2, ... )
  true if OLDER is older than all NEWER.
  Non-exist files are older than any file.

    FileUtils.oldest? ''

--- FileUtils#mkdir( dir1, dir2, ... )
  makes directories DIRs.

    FileUtils.mkdir 'test'
    FileUtils.mkdir 'script', 'text'

--- FileUtils#mkdir_p( dir1, dir2, ... )
  makes dirctories DIRs and all its parent directories.
  For example,

    FileUtils.mkdir_p '/usr/local/bin/ruby'
  
  causes to make directories below (if it is not exist).
      * /usr
      * /usr/local
      * /usr/local/bin
      * /usr/local/bin/ruby

--- FileUtils#ln( old, new )
  makes hard link NEW which links to OLD.
  If NEW is a directory, creates link NEW/OLD.
--- FileUtils#ln( file1, file2 ..., dir )
  links DIR/OLD1 to OLD1, DIR/OLD2 to OLD2, ....

    FileUtils.ln '

--- FileUtils#ln_s( old, new )
  makes symbolic link NEW which links to OLD.
  If last argument is a directory, links DIR/OLD1 to OLD1,
  DIR/OLD2 to OLD2, ....

    File.ln_s 'somefile', 'link_to_somefile'

--- FileUtils#ln_s( file1, file2 ..., dir )
  makes symbolic link dir/file1, dir/file2 ... which point to
  file1, file2 ... If DIR is not a directory, raises Errno::ENOTDIR.

--- FileUtils#cp( src, dest )
  copies a file SRC to DEST. If DEST is a directory, copies
  SRC to DEST/SRC.

    File.cp 'eval.c', 'eval.c.org'

--- FileUtils#cp( file1, file2 ..., dir )
  copies FILE1 to DIR/FILE1, FILE2 to DIR/FILE2 ...

    File.cp 'cgi.rb', 'complex.rb', 'date.rb', '/usr/lib/ruby/1.6'

--- FileUtils#cp_r( src, dest )
  copies SRC to DEST. If SRC is a directory, this method copies
  its all contents recursively. If DEST is a directory, copies
  SRC to DEST/SRC.

    # installing package "mylib" under the site_ruby
    FileUtils.cp_r 'lib/', site_ruby + '/mylib'

--- FileUtils#cp_r( file1, file2 ..., dir )
  copies FILE1 to DIR/FILE1, FILE2 to DIR/FILE2 ...
  If FILE is a directory, copies its all contents recursively.

    FileUtils.cp_r 'mail.rb', 'field.rb', 'debug/', site_ruby + '/tmail'

--- FileUtils#mv( src, dest )
  moves a file SRC to DEST.
  If FILE and DEST exist on the different disk partition,
  copies it.

    FileUtils.mv 'badname.rb', 'goodname.rb'

--- FileUtils#mv( file1, file2 ..., dir )
  moves FILE1 to DIR/FILE1, FILE2 to DIR/FILE2 ...
  If FILE and DEST exist on the different disk partition,
  copies it.

    FileUtils.mv 'junk.rb', 'dust.rb', '.trash/'

--- FileUtils#rm( file1, file2, ... )
  remove FILEs. This method cannot remove directory.

    FileUtils.rm '/usr/bin/perl'   # :-)

--- FileUtils#rm_f( file1, file2, ... )
  remove FILEs. This method ignores all errors which is
  raised in operation.

    FileUtils.rm_f 'junk.rb', 'dust.rb'
    FileUtils.rm_f Dir['*.so'], Dir['*.o']   # nested array is OK.
    FileUtils.rm_f 'NotExistFile'            # no error

--- FileUtils#rm_rf( file1, file2, ... )
  remove FILEs. If FILE is a directory, removes its all contents
  recursively. This method ignores all errors which is raised in
  operation.

    FileUtils.rm_rf '/home'  # if you want to destroy the world

--- FileUtils#cmp( a, b )
--- FileUtils#identical?( a, b )
  true if contents of file A and B is identical.

    FileUtils.cmp 'somefile', 'somefile'  #=> true
    FileUtils.cmp '/bin/cp', '/bin/mv'    #=> maybe false.

--- FileUtils#install( src, dest, mode = <from's> )
  If SRC is not same to DEST, copies it and changes the permittion
  mode to MODE.

    FileUtils.install 'ruby', '/usr/local/bin/ruby', 0755
    FileUtils.install 'lib.rb', '/usr/local/lib/ruby/site_ruby'

--- FileUtils#chmod( mode, file1, file2, ... )
  changes permittion bits on the named FILEs to the bit pattern
  represented by MODE.

    FileUtils.chmod 0644 'my.rb', 'your.rb'
    FileUtils.chmod 0755 'goodcmd'

== module FileUtils::Verbose

This class has all methods of FileUtils::Normal and it works as
same, but outputs messages before action. You can also pass
verbose flag to all methods, as last argument (default is "true").

=== Methods

---- fileutils_op
---- fileutils_op=( operator )

---- fileutils_output
---- fileutils_output=( out )

---- fileutils_label
---- fileutils_label=( str )


== module FileUtils::NoWrite

This class has all methods of FileUtils::Normal but never
changes files.

=end


module FileUtils

  def self.alias_module_function( new, old )
    alias_method new, old
    module_function new
  end


  #
  # FileUtils::Normal
  #

  Normal = self

    module_function


    def cd( dname, &block )
      Dir.chdir dname, &block
    end

    alias_module_function :chdir, :cd

    def pwd
      Dir.pwd
    end

    alias_module_function :getwd, :pwd


    def newest?( new, *fnames )
      return false unless FileTest.exist? new
      new_time = File.ctime(new)
      fnames.each do |old|
        if FileTest.exist? old then
          return false unless new_time > File.mtime(old)
        end
      end
      true
    end

    alias_module_function :newer?, :newest?
    alias_module_function :uptodate?, :newest?


    def mkdir( *args )
      args.flatten.each do |dir|
        Dir.mkdir dir
      end
    end

    def mkdir_p( *args )
      args.flatten.collect {|n| File.expand_path(n) }.each do |dir|
        stack = []
        until FileTest.directory? dir do
          stack.push dir
          dir = File.dirname(dir)
        end
        stack.reverse_each do |dir|
          Dir.mkdir dir
        end
      end
    end

    alias_module_function :mkpath, :mkdir_p
    alias_module_function :makedirs, :mkdir_p


    def ln( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        File.link src, dest
      end
    end

    def ln_s( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        File.symlink src, dest
      end
    end

    def ln_sf( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        rm_f dest
        File.symlink src, dest
      end
    end


    def cp( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        FUPrivate.copy src, dest
      end
    end

    alias_module_function :copy, :cp

    def cp_r( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        if FileTest.directory? src then
          unless FileTest.directory? dest then
            Dir.mkdir dest
          end
          FUPrivate.do_cp_r src, '.', dest
        else
          FUPrivate.copy src, dest
        end
      end
    end


    def mv( *args )
      FUPrivate.each_src_dest( args ) do |src, dest|
        if /djgpp|cygwin|mswin32/ === RUBY_PLATFORM and
           FileTest.file? dest then
          File.unlink dest
        end

        begin
          File.rename src, dest
        rescue
          if FileTest.symlink? src then
            File.symlink File.readlink(src), dest
            File.unlink src
          else
            st = File.stat(src)
            FUPrivate.copy src, dest
            File.unlink src
            File.utime st.atime, st.mtime, dest
            begin
              File.chown st.uid, st.gid, dest
            rescue
              # ignore
            end
          end
        end
      end
    end

    alias_module_function :move, :mv


    def rm( *files )
      files.flatten!
      files.each do |fn|
        File.unlink fn
      end
    end

    alias_module_function :remove, :rm

    def rm_f( *files )
      files.flatten!
      files.each do |fn|
        FUPrivate.rmf fn
      end
    end

    alias_module_function :safe_unlink, :rm_f

    def rm_rf( *files )
      files.flatten!
      files.each do |fn|
        if FileTest.symlink? fn then
          FUPrivate.rmf fn
        elsif FileTest.directory? fn then
          FUPrivate.do_rm_rf fn
        else
          FUPrivate.rmf fn
        end
      end
    end


    def cmp( filea, fileb )
      stra = strb = nil
      size = File.size( filea )
      if File.size( fileb ) != size then
        return true
      end
      if size > FUPrivate::TOO_BIG then
        size = FUPrivate::TOO_BIG
      elsif size < 512 then
        size = 1024
      end

      File.open( filea ) {|a| a.binmode
      File.open( fileb ) {|b| b.binmode
        begin
          while stra == strb do
            stra = a.read( size )
            strb = b.read( size )
            unless stra and strb then
              if stra.nil? and strb.nil? then
                return true
              end
            end
          end
        rescue EOFError
          ;
        end
      } }

      false
    end

    alias_module_function :identical?, :cmp

    def install( from, to, mode = nil )
      dest = FUPrivate.destfn( from, to )
      unless FileTest.exist? dest and cmp( from, dest ) then
        File.chmod 0777, fn
        File.unlink fn
        FUPrivate.copy from, dest
        File.chmod mode, dest if mode
      end
    end

    def update_file( fname, upd )
      str = nil
      if FileTest.file? fname then
        File.open( fname ) {|f| str = f.read }
      end
      if not str or str != upd then
        File.open( fname, 'w' ) do |f|
          f.write upd
        end
      end
    end


    def chmod( mode, *args )
      File.chmod mode, *args.flatten
    end

    def touch( *args )
      t = Time.now
      args.flatten.each do |fname|
        begin
          File.utime(t, t, fname)
        rescue Errno::ENOENT
          File.open(fname, 'a') { }
        end
      end
    end
  
  #end   # module Normal


  module FUPrivate

    TOO_BIG = 2 * 1024 * 1024   # 2 MB

    class << self

    def each_src_dest( args )
      args.flatten!
      case args.size
      when 0, 1
        raise ArgumentError, "too few # of arguments (#{args.size} for >=2)"
      when 2
        from, to = args
        if FileTest.directory? to then
          yield from, (to[-1,1] == '/' ? to : to + '/') + File.basename(from)
        else
          yield args
        end
      else
        dir = args.pop
        # FileTest.directory? dir or raise ArgumentError, "must be dir: #{dir}"
        sep = (dir[-1,1] == '/') ? '' : '/'
        args.each do |fn|
          yield fn, dir + sep + File.basename(fn)
        end
      end
    end

    def destfn( from, to )
      if FileTest.directory? to then
        (to[-1,1] == '/' ? to : to + '/') + File.basename(from)
      else
        to
      end
    end


    def copy( from, to )
      mod = File.stat( from ).mode
      size = s = w = nil

      File.open( from )    {|rf| rf.binmode
      File.open( to, 'w' ) {|wf| wf.binmode
        size = File.size( from )
        if size > TOO_BIG then
          size = TOO_BIG
        elsif size < 512 then
          size = 1024
        end
        s = nil

        begin
          while true do
            s = rf.sysread( size )
            w = wf.syswrite( s )
            while w < s.size do
              w += wf.syswrite( s[ w, s.size - w ] )
            end
          end
        rescue EOFError
          File.chmod mod, to
        end
      } }
    end

    def do_cp_r( base, abs, to )
      Dir.entries( "#{base}/#{abs}" ).each do |fn|
        if FileTest.directory? fn then
          next if /\A\.\.?\z/ === fn
          Dir.mkdir "#{to}/#{abs}/#{fn}"
          do_cp_r base, "#{abs}/#{fn}", to
        else
          copy "#{base}/#{abs}/#{fn}", "#{to}/#{abs}/#{fn}"
        end
      end
    end

    def rmf( fn )
      first = true
      begin
        File.unlink fn
      rescue
        begin
          if first then
            first = false
            File.chmod 0777, fn
            retry
          end
        rescue
        end
      end
    end

    def do_rm_rf( dn )
      Dir.foreach( dn ) do |fn|
        next if /\A\.\.?\z/ === fn
        fn = "#{dn}/#{fn}"
        if FileTest.directory? fn then
          do_rm_rf fn
        else
          rmf fn
        end
      end
      Dir.rmdir dn
    end

    def getflag( arr )
      arr.flatten!
      if arr.last == true or arr.last == false then
        arr.pop
      else
        true
      end
    end

  end; end


  #
  # FileUtils::Verbose
  #

  module Verbose

    @fileutils_op     = FileUtils::Normal
    @fileutils_output = $stderr
    @fileutils_label  = 'fileutils.'

    attr_accessor :fileutils_op
    attr_accessor :fileutils_output
    attr_accessor :fileutils_label

    def futilsmsg( msg )
      @fileutils_op     ||= Normal
      @fileutils_label  ||= 'fileutils.'
      @fileutils_output ||= $stderr
      @fileutils_output.puts @fileutils_label + msg
    end
    module_function :futilsmsg
    private_class_method :futilsmsg


    class << self

      private

      def delegate( name )
        name = name.id2name
        if name[-1,1] != '?' then
          outname = name.sub( '_', ' -' )
        else
          outname = name
        end

        module_eval %^
        def #{name}( *args )
          futilsmsg "#{outname} \#{args.join ' '}" if FUPrivate.getflag(args)
          @fileutils_op.#{name}( *args )
        end
        ^
        module_function name
      end

      def alias_module_function( new, old )
        alias_method new, old
        module_function new
      end

    end


    module_function


    def cd( dirname, verbose = true, &block )
      futilsmsg "cd #{dirname}" if verbose
      @fileutils_op.cd dirname, &block
      futilsmsg "cd #{Dir.pwd}" if verbose and block
    end

    alias_module_function :chdir, :cd

    def pwd( verbose = true )
      futilsmsg 'pwd' if verbose
      Dir.pwd
    end

    alias_module_function :getwd, :pwd


    delegate :newest?
    alias_module_function :uptodate?, :newest?

    delegate :mkdir
    delegate :mkdir_p
    alias_module_function :mkpath, :mkdir_p
    alias_module_function :makedirs, :mkdir_p
    delegate :rmdir

    delegate :ln
    delegate :ln_s
    delegate :ln_sf

    delegate :cp
    alias_module_function :copy, :cp
    delegate :cp_r
    delegate :mv
    alias_module_function :move, :mv
    delegate :rm
    delegate :rm_f
    alias_module_function :safe_unlink, :rm_f
    delegate :rm_rf

    def cmp( a, b, verbose = true )
      futilsmsg "cmp #{a} #{b}" if verbose
      @fileutils_op.cmp a, b
    end

    alias_module_function :compare, :cmp
    alias_module_function :identical?, :cmp

    def install( from, to, mode = nil, verbose = true )
      futilsmsg "install #{mode ? '%o ' % mode : ''}#{from} #{to}" if verbose
      @fileutils_op.install from, to, mode
    end

    def update_file( fname, upd, verbose = true )
      futilsmsg "update #{to}" if verbose
      @fileutils_op.update_file fname, upd
    end


    def chmod( mode, *args )
      futilsmsg sprintf('chmod %o %s',
                        mode, args.join(' ')) if FUPrivate.getflag(args)
      @fileutils_op.chmod mode, *args
    end

    delegate :touch

  end   # module Verbose


  class VerboseDecorator

    include Verbose

    def initialize( op = nil )
      @fileutils_op = op || Normal
      @fileutils_output = $stderr
      @fileutils_label  = 'fileutils.'
    end
  
  end


  #
  # FileUtils::NoWrite
  #

  module NoWrite

    extend FileTest
    include FileTest

    class << self

      private

      def delegate( fname, exec )
        fname = fname.id2name
        str = %-
          def #{fname}( *args, &block )
          #{if exec then
              'FileUtils::Normal.' + fname + '( *args, &block )'
            else
              ''
            end}
          end
          module_function :#{fname}
        -
        module_eval str, __FILE__, @line
        @line += 1
      end

    end

    @line = __LINE__
    delegate :cd           ,true
    delegate :pwd          ,true
    delegate :getwd        ,true
    delegate :newest?      ,true
    delegate :uptodate?    ,true
    delegate :mkdir        ,false
    delegate :mkdir_p      ,false
    delegate :makedirs     ,false
    delegate :mkpath       ,false
    delegate :ln           ,false
    delegate :ln_s         ,false
    delegate :cp           ,false
    delegate :copy         ,false
    delegate :cp_r         ,false
    delegate :mv           ,false
    delegate :move         ,false
    delegate :rm           ,false
    delegate :rm_f         ,false
    delegate :rm_rf        ,false
    delegate :cmp          ,true
    delegate :compare      ,true
    delegate :identical?   ,true
    delegate :install      ,false
    delegate :update_file  ,false
    delegate :chmod        ,false
    delegate :touch        ,false
    remove_instance_variable :@line

  end   # module NoWrite

end
