#
# encode.rb
#
#   Copyright (c) 1999,2000 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'nkf'

require 'strscan'
require 'amstd/bug'


module TMail

  class HFdecoder

    def initialize( ret = nil, charset = nil )
      @f     = ret || ''
      @opt   = '-' + (/ejs/ === charset ? charset : 'e')
    end

    class << self
      def decode( str, outcode = 'e' )
        NKF.nkf( "-#{outcode}m", str )
      end
    end


    def write_in
      @f
    end

    def header_line( str )
      @f << decode( str )
    end

    def header_name( nm )
      @f << nm << ': '
    end

    def header_body( str )
      @f << decode( str )
    end
      
    def space
      @f << ' '
    end

    alias spc space

    def lwsp( str )
      @f << str
    end
      
    def meta( str )
      @f << str
    end

    def text( str )
      @f << decode( str )
    end

    alias phrase text
    alias value text


    private

    def decode( str )
      NKF.nkf( @opt + 'm', str )
    end

  end


  class HFencoder

    unless defined? BENCODE_DEBUG then
      BENCODE_DEBUG = false
    end


    class << self
      def encode( str )
        ret = ''
        e = new( ret )
        e.header_body( str )
        e.write_in
        ret
      end
    end


    def initialize( ret = nil, eol = nil, charset = nil, limit = nil )
      @f     = ret || ''
      @eol   = eol || "\r\n"
      @space = ' '
      @limit = (limit || 72) - @eol.size
      @opt   = '-' + case charset
                     when 'e', 'j', 's' then charset
                     else                    'j'
                     end
      reset
    end

    def reset
      @buf = ''
      @lwsp = ''
      @linelen = 0
    end


    ###
    ### add
    ###

    def header_line( line )
      scanadd line
    end

    def header_name( name )
      add_text name.split('-').collect {|i| i.capitalize }.join('-')
      add_text ':'
      add_lwsp ' '
    end

    def header_body( str )
      scanadd NKF.nkf( @opt, str )
    end

    def space
      add_lwsp ' '
    end

    alias spc space

    def lwsp( str )
      add_lwsp str.sub( /[\r\n]+[^\r\n]*\z/, '' )
    end

    def meta( str )
      add_text str
    end

    def text( str )
      scanadd NKF.nkf( @opt, str )
    end

    def phrase( str )
      str = NKF.nkf( @opt, str )
      if CONTROL === str then
        scanadd str
      else
        add_text TMail.quote_phrase( str )
      end
    end

    def value( str )
      str = NKF.nkf( @opt, str )
      if CONTROL === str then
        encodeadd str
      else
        add_text TMail.quote(str)
      end
    end

    ###
    ### terminate
    ###

    def write_in
      terminate
      reset
      @f
    end


    private


    MPREFIX = '=?iso-2022-jp?B?'
    MSUFFIX = '?='

    ESC_ASCII     = "\e(B"
    ESC_ISO2022JP = "\e$B"


    def scanadd( str, force = false )
      types = ''
      strs = []
      pres = prev = nil

      s = StringScanner.new( str, false )

      until s.empty? do
        if tmp = s.scan( /\A[^\e\t\r\n ]+/ ) then
          types <<  pres = force ? 'j' : 'a'
          strs.push prev = tmp

        elsif tmp = s.scan( /\A[\t\r\n ]+/ ) then
          types <<  pres = 's'
          strs.push prev = tmp

        elsif esc = s.scan( /\A\e../ ) then
          if esc != ESC_ASCII and tmp = s.scan( /\A[^\e]+/ ) then
            types <<  pres = 'j'
            strs.push prev = tmp
          end

        else
          bug! "HFencoder#scan, not match"
        end
      end

      do_encode types, strs
    end

    def do_encode( types, strs )
      #
      # sbuf = (J|A)(J|A|S)*(J|A)
      #
      #   A: ascii only, no need to encode
      #   J: jis, etc. need to encode
      #   S: LWSP
      #
      # (J|A)*J(J|A)* -> W
      # W(SW)*        -> E
      #
      # result = (A|E)(S(A|E))*
      #
      if BENCODE_DEBUG then
        puts "\n-- do_encode ------------"
        puts types.split(//).join(' ')
        p strs
      end

      e = /[ja]*j[ja]*(?:s[ja]*j[ja]*)*/

      while m = e.match(types) do
        pre = m.pre_match
        unless pre.empty? then
          concat_a_s pre, strs[ 0, pre.size ]
        end

        concat_e m[0], strs[ m.begin(0)..m.end(0) ]

        types = m.post_match
        strs.slice! 0, m.end(0)
      end
      concat_a_s types, strs
    end

    def concat_a_s( types, strs )
      i = 0
      types.each_byte do |t|
        case t
        when ?a then add_text strs[i]
        when ?s then add_lwsp strs[i]
        else
          bug!
        end
        i += 1
      end
    end
    
    def concat_e( types, strs )
      if BENCODE_DEBUG then
        puts '---- concat_e'
        puts "types=#{types.split('').join(' ')}"
        puts "strs =#{strs.inspect}"
      end

      bug! @buf unless @buf.empty?

      chunk = ''
      strs.each_with_index do |s,i|
        m = 'extract_' + types[i,1]
        until s.empty? do
          unless c = send(m, chunk.size, s) then
            add_with_encode chunk unless chunk.empty?
            flush
            chunk = ''
            fold
            c = send( m, 0, s )
            bug! unless c
          end
          chunk << c
        end
      end
      add_with_encode chunk unless chunk.empty?
    end

    def extract_j( csize, str )
      size = maxbyte( csize, str.size ) - 6
      size = size % 2 == 0 ? size : size - 1
      return nil if size <= 0

      c = str[ 0, size ]
      str[ 0, size ] = ''
      c = ESC_ISO2022JP + c + ESC_ASCII
      c
    end

    def extract_a( csize, str )
      size = maxbyte( csize, str.size )
      return nil if size <= 0

      c = str[ 0, size ]
      str[ 0, size ] = ''
      c
    end

    alias extract_s extract_a

    def maxbyte( csize, ssize )
      rest = restsize - MPREFIX.size - MSUFFIX.size
      rest / 4 * 3 - csize
    end

    #def encsize( len )
    #  amari = (if len % 3 == 0 then 0 else 1 end)
    #  (len / 3 + amari) * 4 + PRESIZE
    #end


    ###
    ### length-free buffer
    ###

    def add_text( str )
# puts "text: #{str.inspect}"
      @buf << str
    end

    def add_with_encode( str )
# puts "enco: #{str.inspect}"
      s = [str].pack('m')
      s.chop!
      @buf << MPREFIX << s << MSUFFIX
    end

    def add_lwsp( lwsp )
# puts "lwsp: #{lwsp.inspect}"
      if restsize <= 0 then
        fold
        flush
      else
        flush
        @lwsp = lwsp
      end
    end

    def flush
      @f << @lwsp << @buf
      @linelen += @lwsp.size + @buf.size
      @buf = ''
      @lwsp = ''
    end

    def fold
      @f << @eol
      @linelen = 0
      @lwsp = @space
    end

    def terminate
      add_lwsp ''
    end

    def restsize
      @limit - (@linelen + @lwsp.size + @buf.size)
    end

  end   # class HFencoder

end    # module TMail
