#
# setup.rb version 1.0.7
#
#   Copyright (c) 2000 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'tempfile'

$:.unshift './lib'
require 'amstd/futils'
require 'amstd/rbparams'
require 'amstd/bug'


class Installer

  extend FileUtils
  include FileUtils

  #
  # name => [ default, argument-name, discription ]
  #
  OPTIONS = {
    'bin-dir'   => [ RubyParams::BINDIR,
                     'path',
                     'directory for binary' ],
    'rb-dir'    => [ RubyParams::SITE_RB,
                     'path',
                     'directory for ruby script' ],
    'so-dir'    => [ RubyParams::SITE_SO,
                     'path',
                     'directory for ruby extention' ],
    'data-dir'  => [ RubyParams::DATADIR,
                     'path',
                     'directory for data' ],
    'ruby-path' => [ RubyParams::RUBY_PATH,
                     'path',
                     'path to ruby interpreter' ],
    'make-prog' => [ 'make',
                     'name',
                     'make program for ruby extention' ],
    'with'      => [ '',
                     'name,name...',
                     'package name(s) you want to install' ],
    'without'   => [ '',
                     'name,name...',
                     'package name(s) you do not want to install' ]
  }

  OPTION_ORDER = %w( bin-dir rb-dir so-dir ruby-path make-prog with without )

  TASKS = {
    'config'       => 'set config option',
    'setup'        => 'compile extention or else',
    'install'      => 'install packages',
    'clean'        => "do `make clean' for each extention",
    'dryrun'       => 'test run',
    'show'         => 'show current configuration'
  }

  TASK_ORDER = %w( config setup install clean dryrun show )

  TYPES = %w( bin lib ext share )


  def initialize( argv )
    @verbose = true
    @config = {}
    @task = nil
    @other_args = []

    configure argv
  end

  attr :config
  attr :task


  ConfigFile = 'config.save'

  def configure( argv )
    argv = argv.dup
    OPTIONS.each do |k,v|
      @config[ k ] = v[0]
    end
    if exist? ConfigFile then
      read_config
    end

    exp = /\A--(#{OPTIONS.keys.join '|'})=/
    tasks = /\A(?:#{TASKS.keys.join '|'})\z/

    wrong_args = []
    while i = argv.shift do
      case i
      when exp
        name = $1
        ar   = $'.strip

        unless @config[ name ] then
          raise ArgumentError, "unknown argument '--#{name}'"
        end
        @config[ name ] = ar

      when tasks
        if @task then
          raise ArgumentError, "too many # of task: '#{@task}' and '#{i}'"
        end
        @task = i

      when '-h', '--help'
        print_usage $stdout
        exit 0

      when '--'
        @other_args = argv
        break

      else
        wrong_args.push i
        false
      end
    end

    unless wrong_args.empty? then
      raise ArgumentError, "unknown option '#{wrong_args.join %<', '>}'"
    end
    unless @task then
      raise ArgumentError, 'task not given'
    end
    check_packdesig
  end

  def write_config
    File.open( ConfigFile, 'w' ) do |f|
      @config.each do |k,v|
        f.printf "%s=%s\n", k, v if v
      end
    end
  end

  def read_config
    File.foreach( ConfigFile ) do |line|
      k, v = line.split( '=', 2 )
      @config[ k.strip ] = v.strip
    end
  end

  def print_usage( out )
    out.puts
    out.puts 'Usage:'
    out.puts '  ruby setup.rb <task> [<options>]'
    out.puts
    out.puts 'Tasks:'
    TASK_ORDER.each do |name|
      out.printf "  %-10s  %s\n", name, TASKS[name]
    end
    out.puts
    out.puts 'Config Options:'
    OPTION_ORDER.each do |name|
      dflt, arg, desc = OPTIONS[name]
      out.printf "  %-20s %s [%s]\n", "--#{name}=#{arg}", desc, dflt
    end
    out.puts
    out.puts 'This archive includes:'
    out.print '  ', packages().join(' '), "\n"
    out.puts
  end


  def execute
    case @task
    when 'config', 'setup', 'install', 'clean'
      tryto @task
    when 'show'
      do_show
    when 'dryrun'
      do_dryrun
    else
      bug!
    end
  end

  def tryto( task )
    $stderr.printf "entering %s phase...\n", task
    begin
      send 'do_' + task
    rescue
      $stderr.printf "%s failed\n", task
      raise
    end
    $stderr.printf "%s done.\n", task
  end

  def do_config
    write_config
  end

  def do_show
    OPTION_ORDER.each do |k|
      v = @config[k]
      if not v or v.empty? then
        v = '(not specify)'
      end
      printf "%-10s %s\n", k, v
    end
  end

  def do_setup
    into_dir( 'bin' ) {
      foreach_package do
        Dir.foreach( '.' ) do |fname|
          next unless File.file? fname
          add_rubypath fname
        end
      end
    }
    into_dir( 'ext' ) {
      foreach_package do
        extconf
        make
      end
    }
  end

  def do_install
    into_dir( 'bin' ) {
      foreach_package do |dn, targ|
        install_bin
      end
    }
    into_dir( 'lib' ) {
      foreach_package do |dn, targ|
        install_rb targ
      end
    }
    into_dir( 'ext' ) {
      foreach_package do |dn, targ|
        install_so targ
      end
    }
    into_dir( 'share' ) {
      foreach_package do |dn, targ|
        install_dat targ
      end
    }
  end

  def do_clean
    into_dir( 'ext' ) {
      foreach_package do |dn, targ|
        clean
      end
    }
  end
  
  def do_dryrun
    unless exist? 'tmp' then
      $stderr.puts 'setup.rb: setting up temporaly environment...'
      @verbose = $DEBUG
      begin
        @config['bin-dir']  = isdir(expand('.'), 'tmp', 'bin')
        @config['rb-dir']   = isdir(expand('.'), 'tmp', 'lib')
        @config['so-dir']   = isdir(expand('.'), 'tmp', 'ext')
        @config['data-dir'] = isdir(expand('.'), 'tmp', 'share')
        do_install
      rescue
        rm_rf 'tmp'
        $stderr.puts '[BUG] setup.rb bug: "dryrun" command failed'
        raise
      end
    end

    exec @config['ruby-path'],
         '-I' + fjoin('.', 'tmp', 'lib'),
         '-I' + fjoin('.', 'tmp', 'ext'),
         *@other_args
  end
  
  #
  # lib
  #

  def into_dir( libn )
    return unless directory? libn
    chdir( libn ) do
      yield
    end
  end


  def check_packdesig
    @with = extract_dirs( @config['with'] )
    @without = extract_dirs( @config['without'] )

    packs = packages
    (@with + @without).each do |i|
      if not packs.include? i and not directory? i then
        raise ArgumentError, "no such package or directory '#{i}'"
      end
    end
  end

  def extract_dirs( s )
    ret = []
    s.split(',').each do |i|
      if /[\*\?]/ === i then
        tmp = Dir.glob(i)
        tmp.delete_if {|d| not File.directory? d }
        if tmp.empty? then
          tmp.push i   # causes error
        else
          ret.concat tmp
        end
      else
        ret.push i
      end
    end

    ret
  end

  def foreach_record( fn = 'PATHCONV' )
    File.foreach( fn ) do |line|
      line.strip!
      next if line.empty?
      a = line.split(/\s+/, 3)
      a[2] ||= '.'
      yield a
    end
  end

  def packages
    ret = []
    TYPES.each do |type|
      next unless File.exist? type
      foreach_record( "#{type}/PATHCONV") do |dir, pack, targ|
        ret.push pack
      end
    end
    ret.uniq
  end

  def foreach_package
    path = {}
    foreach_record do |dir, *rest|
      path[dir] = rest
    end

    base = basename( Dir.getwd )
    Dir.foreach('.') do |dir|
      next if dir[0] == ?.
      next unless directory? dir

      unless path[dir] then
        bug! "abs path for package '#{dir}' not exist"
      end
      pack, targ = path[dir]

      if inclpack( pack, "#{base}/#{dir}" ) then
        chdir( dir ) {
          yield dir, targ
        }
      else
        $stderr.puts "setup.rb: skip #{base}/#{dir}(#{pack}) by user option"
      end
    end
  end

  def inclpack( pack, dname )
    if @with.empty? then
      not @without.include? pack and
      not @without.include? dname
    else
      @with.include? pack or
      @with.include? dname
    end
  end


  def add_rubypath( fn, opt = nil )
    line = "\#!#{@config['ruby-path']}#{opt ? ' ' + opt : ''}"

    $stderr.puts %Q<setting #! line to "#{line}"> if @verbose

    tmpf = nil
    File.open( fn ) do |f|
      first = f.gets
      return unless /\A\#\!.*ruby/ === first

      tmpf = Tempfile.open( 'amsetup' )
      tmpf.puts line
      tmpf << first unless /\A\#\!/o === first
      f.each {|i| tmpf << i }
      tmpf.close
    end
    
    mod = File.stat( fn ).mode
    tmpf.open
    File.open( fn, 'w' ) do |wf|
      tmpf.each {|i| wf << i }
    end
    chmod mod, fn

    tmpf.close true
  end


  def install_bin
    install_all isdir( @config['bin-dir'] ), 0555
  end

  def install_rb( dir )
    install_all isdir(fjoin(@config['rb-dir'], dir)), 0644
  end

  def install_dat( dir )
    install_all isdir(fjoin(@config['data-dir'], dir)), 0644
  end

  def install_all( dir, mode )
    Dir.foreach('.') do |fname|
      next if /\A\./ === fname
      next unless File.file? fname

      install fname, dir, mode
    end
  end


  def extconf
    system "#{@config['ruby-path']} extconf.rb"
  end

  def make
    command @config['make-prog']
  end
  
  def clean
    command @config['make-prog'] + ' clean'
  end

  def install_so( dir )
    to = isdir(fjoin(@config['so-dir'], dir))
    find_so('.').each do |fn|
      install fn, to, 0555
    end
  end

  def find_so( dir = '.' )
    fnames = nil
    Dir.open( dir ) {|d| fnames = d.to_a }
    exp = /\.#{RubyParams::DLEXT}\z/
    arr = fnames.find_all {|fn| exp === fn }
    unless arr then
      raise ArgumentError,
        "can't find ruby extention library in '#{expand dir}'"
    end

    arr
  end

  def so_dir?( dn = '.' )
    file? fjoin( dn, 'MANIFEST' )
  end

end


begin
  MainInstaller = Installer.new( ARGV )
  MainInstaller.execute
rescue
  raise if $DEBUG
  $stderr.puts $!
  $stderr.puts 'try "ruby setup.rb --help" for usage'
  exit 1
end
