#
# setup.rb
#
#   Copyright (c) 2000 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#

require 'tempfile'

$:.unshift './lib'
require 'amstd/futils'
require 'amstd/rbparams'
require 'amstd/bug'


class Installer

  extend FileUtils
  include FileUtils

  #
  # name => [ default, argument, discription ]
  #
  Options = {
    'bin-dir'   => [ RubyParams::BINDIR,
                     'path',
                     'path for binary' ],
    'rb-dir'    => [ RubyParams::SITE_RB,
                     'path',
                     'path for ruby script' ],
    'so-dir'    => [ RubyParams::SITE_SO,
                     'path',
                     'path for ruby extention' ],
    'ruby-path' => [ RubyParams::RUBY_PATH,
                     'path',
                     'path of ruby interpreter' ],
    'with'      => [ '',
                     'name,name...',
                     'package name(s) you want to install' ],
    'without'   => [ '',
                     'name,name...',
                     'package name(s) you do not want to install' ]
  }

  OptionOrder = %w( bin-dir rb-dir so-dir ruby-path with without )

  Tasks = {
    'config'       => 'set config option',
    'setup'        => 'compile extention or else',
    'install'      => 'install packages',
    'show'         => 'show current configuration'
  }

  TaskOrder = %w( config setup install show )

  def initialize( argv )
    @verbose = true
    @config = {}
    @task = nil
    @wrong_arg = []

    configure argv
  end

  attr :config
  attr :task
  attr :wrong_arg


  ConfigFile = 'config.save'

  def configure( argv )
    Options.each do |k,v|
      @config[ k ] = v[0]
    end
    if exist? ConfigFile then
      read_config
    end

    exp = /\A--(#{Options.keys.join '|'})=/
    tasks = /\A(?:#{Tasks.keys.join '|'})\z/

    wrong_arg = []
    argv.delete_if do |i|
      case i
      when exp
        name = $1
        ar   = $'.strip

        unless @config[ name ] then
          raise ArgumentError, "unknown argument '--#{name}'"
        end
        @config[ name ] = ar
        true

      when tasks
        if @task then
          raise ArgumentError, "too many # of task: '#{@task}' and '#{i}'"
        end
        @task = i
        true

      when '-h', '--help'
        print_usage $stdout
        exit 0

      else
        wrong_arg.push i.dup
        false
      end
    end

    unless wrong_arg.empty? then
      raise ArgumentError, "unknown option '#{wrong_arg.join %<', '>}'"
    end
    unless @task then
      raise ArgumentError, 'task not given'
    end
  end

  def write_config
    File.open( ConfigFile, 'w' ) do |f|
      @config.each do |k,v|
        f.printf "%s=%s\n", k, v if v
      end
    end
  end

  def read_config
    File.foreach( ConfigFile ) do |line|
      k, v = line.split( '=', 2 )
      @config[ k.strip ] = v.strip
    end
  end

  def print_usage( out )
    out.puts
    out.puts 'Usage:'
    out.puts '  ruby setup.rb <task> [<options>]'
    out.puts
    out.puts 'Tasks:'
    TaskOrder.each do |name|
      out.printf "  %-10s  %s\n", name, Tasks[name]
    end
    out.puts
    out.puts 'Config Options:'
    OptionOrder.each do |name|
      dflt, arg, desc = *Options[name]
      out.printf "  %-20s %s [%s]\n", "--#{name}=#{arg}", desc, dflt
    end
    out.puts
    out.puts 'This archive includes:'
    out.print '  ', packages().join(' '), "\n"
    out.puts
  end


  def execute
    case @task
    when 'config', 'setup', 'install'
      tryto @task
    when 'show'
      do_show
    else
      bug!
    end
  end

  def tryto( task )
    $stderr.printf "entering %s phase...\n", task
    begin
      send 'do_' + task
    rescue
      $stderr.printf "%s failed\n", task
      raise
    end
    $stderr.printf "%s done.\n", task
  end

  def do_config
    write_config
  end

  def do_show
    OptionOrder.each do |k|
      v = @config[k]
      if not v or v.empty? then
        v = '(not specify)'
      end
      printf "%-10s %s\n", k, v
    end
  end

  def do_setup
    into_dir( 'bin' ) {
      foreach_package do
        Dir.foreach( '.' ) do |fname|
          next unless File.file? fname
          add_rubypath fname
        end
      end
    }
    into_dir( 'ext' ) {
      foreach_package do
        extconf
        make
      end
    }
  end

  def do_install
    into_dir( 'bin' ) {
      foreach_package do |pn, dir|
        install_bin
      end
    }
    into_dir( 'lib' ) {
      foreach_package do |pn, dir|
        install_rb dir
      end
    }
    into_dir( 'ext' ) {
      foreach_package do |pn, dir|
        install_so dir
      end
    }
  end


  #
  # lib
  #

  def into_dir( libn )
    return unless directory? libn
    chdir( libn ) do
      yield
    end
  end


  def packages
    ret = []
    %w( bin lib ext share ).each do |type|
      fn = fjoin(type, 'PATHCONV')
      next unless File.exist? fn
      File.foreach( fn ) do |line|
        ret.push line.strip.split(' ', 3)[1]
      end
    end
    ret.uniq
  end

  def foreach_package
    with = @config['with'].split(',')
    without = @config['without'].split(',')

    path = {}
    File.foreach( 'PATHCONV' ) do |line|
      nm, pack, dir = *line.strip.split(' ', 3)
      if not dir or dir.empty? then
        dir = '.'
      end
      path[nm] = [pack, dir]
    end

    Dir.foreach( '.' ) do |fn|
      next if /\A\./o === fn
      next unless File.directory? fn

      unless path[fn] then
        bug! "abs path for package '#{fn}' not exist"
      end
      pack, dir = *path[fn]

      if inclpack( pack, with, without ) then
        chdir( fn ) {
          yield fn, dir
        }
      else
        $stderr.puts "setup.rb: skip package '#{pack}' (user option)"
      end
    end
  end

  def inclpack( pack, with, without )
    if with.empty? then
      not without.include? pack
    else
      with.include? pack
    end
  end


  def add_rubypath( fn, opt = '' )
    opt = ' ' + opt unless opt.empty?

    tmpf = nil
    File.open( fn ) do |f|
      first = f.gets
      return unless /\A\#\!/ === first

      tmpf = Tempfile.open( 'amsetup' )
      tmpf.puts "\#!#{@config['ruby-path']}#{opt}"
      tmpf << first unless /\A\#\!/o === first
      f.each {|i| tmpf << i }
      tmpf.close
    end
    
    mod = File.stat( fn ).mode
    tmpf.open
    File.open( fn, 'w' ) do |wf|
      tmpf.each {|i| wf << i }
    end
    chmod mod, fn

    tmpf.close true
  end


  def install_bin
    install_all isdir( @config['bin-dir'] ), 0555
  end

  def install_rb( dir )
    install_all isdir(fjoin(@config['rb-dir'], dir)), 0644
  end

  def install_all( dir, mode )
    Dir.foreach('.') do |fname|
      next if /\A\./ === fname
      next unless File.file? fname

      install fname, dir, mode
    end
  end


  def extconf
    unless is_newer? 'Makefile', 'extconf.rb' then
      system "#{@config['ruby-path']} extconf.rb"
    end
  end

  def make
    command 'make'
  end

  def install_so( dir )
    to = isdir(fjoin(@config['so-dir'], dir))
    find_so('.').each do |fn|
      install fn, to, 0555
    end
  end

  def find_so( dir = '.' )
    fnames = nil
    Dir.open( dir ) {|d| fnames = d.to_a }
    exp = /\.#{RubyParams::DLEXT}\z/
    arr = fnames.find_all {|fn| exp === fn }
    unless arr then
      raise ArgumentError,
        "can't find ruby extention library in '#{expand dir}'"
    end

    arr
  end

  def so_dir?( dn = '.' )
    file? fjoin( dn, 'MANIFEST' )
  end

end


begin
  MainInstaller = Installer.new( ARGV.dup )
  MainInstaller.execute
rescue
  raise if $DEBUG
  $stderr.puts $!
  $stderr.puts 'try "ruby setup.rb --help" for usage'
  exit 1
end
