#
# encode.rb
#
#   Copyright (c) 1999,2000 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'nkf'

require 'strscan'
require 'amstd/bug'


module TMail


class HFdecoder

  def initialize( ret = nil, charset = nil )
    @f     = ret || ''
    @opt   = '-' + (/ejs/ === charset ? charset : 'e')
  end

  class << self
    def decode( str, outcode = 'e' )
      NKF.nkf( "-#{outcode}m", str )
    end
  end


  def write_in
    @f
  end

  def header_line( str )
    @f << decode( str )
  end

  def header_name( nm )
    @f << nm << ': '
  end

  def header_body( str )
    @f << decode( str )
  end
    
  def spc
    @f << ' '
  end

  def lwsp( str )
    @f << str
  end
    
  def meta( str )
    @f << str
  end

  def text( str )
    @f << decode( str )
  end

  alias phrase text
  alias value text


  private

  def decode( str )
    NKF.nkf( @opt + 'm', str )
  end

end


class HFencoder

  unless defined? BENCODE_DEBUG then
    BENCODE_DEBUG = false
  end

  def initialize( ret = nil, eol = nil, charset = nil, limit = nil )
    @f     = ret   || ''
    @sep   = (eol  || "\r\n") + ' '
    @limit = limit || 72
    @opt   = '-' + case charset
                   when 'e', 'j', 's' then charset
                   else                    'j'
                   end
    @len      = @limit - @sep.size
    @chunkmax = chunk_max( @len )

    reset
  end

  class << self
    def encode( str )
      ret = ''
      e = new( ret )
      e.header_body( str )
      e.write_in
      ret
    end
  end


  def reset
    @buf = ''
    @lwsp = ''
    @width = 0
    @first_flush = true
  end

  def write_in
    flush ''
    reset
    @f
  end

  def header_line( line )
    scanadd line
  end

  def header_name( nm )
    @buf << nm << ': '
  end

  def header_body( str )
    scanadd NKF.nkf( @opt, str )
  end

  def spc
    flush ' '
  end

  def lwsp( str )
    str.sub!( /[\r\n]+[^\r\n]*\z/, '' )
    flush str
  end

  def meta( str )
    @buf << str
  end

  def text( str )
    scanadd NKF.nkf( @opt, str )
  end

  def phrase( str )
    str = NKF.nkf( @opt, str )
    if CONTROL === str then
      scanadd str
    else
      @buf << TMail.quote_phrase( str )
    end
  end

  def value( str )
    str = NKF.nkf( @opt, str )
    if CONTROL === str then
      s = [str].pack('m')
      s.chop!
      @buf << MPREFIX << s << MSUFFIX
    else
      @buf << TMail.quote(str)
    end
  end


  private


  def add( str )
    @f << str
    @width += str.size
  end

  def add_with_lf( str )
    if @width > @len then
      @f << @eol
      @width = 0
    end
    add str
  end


  def flush( next_lwsp, force = false )
    puts "flush: #{force ? '(force) ' : ''}next_lwsp=#{next_lwsp.inspect}" if BENCODE_DEBUG

    lwsp = nil

    if @first_flush then
      @first_flush = false
      lwsp = ''
      @width = @buf.size
      puts "flush: not fold, buf=#{@buf.inspect}" if BENCODE_DEBUG
    else
      w = @width + @lwsp.size + @buf.size
      if w > @len then
        puts "flush: fold line" if BENCODE_DEBUG
        lwsp = @sep
        @width = @buf.size
      else
        puts "flush: not fold, buf=#{@buf.inspect}" if BENCODE_DEBUG
        lwsp = @lwsp
        @width = w
      end
    end

    @f << lwsp; @lwsp = next_lwsp
    @f << @buf; @buf = ''
  end


  ESC_ASCII = "\e(B"

  def scanadd( str, force = false )
    sbuf = []
    vbuf = []
    tmp = pre = i = nil
    s = StringScanner.new( str, false )

    while s.rest? do
      if tmp = s.scan( /\A[^\e\t\r\n ]+/ ) then
        sbuf.push force ? :J : :A
        vbuf.push tmp

      elsif tmp = s.scan( /\A[\t\r\n ]+/ ) then
        sbuf.push :S
        vbuf.push tmp

      elsif pre = s.scan( /\A\e../ ) then
        next if pre == ESC_ASCII

## faster
#while tmp = s.scan( /\A[^\e]{1,30}/ ) do
#  sbuf.push :J
#  vbuf.push tmp
#end
        if tmp = s.scan( /\A[^\e]+/ ) then
          0.step( tmp.size, @chunkmax ) do |i|
            sbuf.push :J
            vbuf.push pre + tmp[ i, @chunkmax ] + ESC_ASCII
          end
        end

      else
        bug! "HFencoder#scan, not match"
      end
    end

    do_encode( sbuf, vbuf )
  end


  def do_encode( sbuf, vbuf )
    #
    # sbuf = (J|A)(J|A|S)*(J|A)
    #
    #   A: ascii only, no need to encode
    #   J: jis, etc. need to encode
    #   S: LWSP
    #
    # (J|A)*J(J|A)* -> W
    # W(SW)*        -> E
    #
    # result = (A|E)(S(A|E))*
    #
    if BENCODE_DEBUG then
      puts "\n-- parse_words ------------\n\n"
      puts sbuf.collect{|i| i.id2name }.inspect
      puts vbuf.inspect
    end

    beg = i = endi = si = 0
    t = nil
    while sbuf[i] do
      while (t = sbuf[i]) and t == :A do i += 1 end

      if sbuf[i] == :J then
        #
        #  E
        #
        while (t = sbuf[i]) and t != :S do i += 1 end
        endi = i

        while true do
          while (t = sbuf[i]) and t == :S do i += 1 end
          si = i
          while (t = sbuf[i]) and t == :A do i += 1 end
          break unless sbuf[i] == :J
          while (t = sbuf[i]) and t != :S do i += 1 end

          endi = i
        end
        concat_e vbuf[ beg, endi - beg ]

        if i > endi then
          flush ' ' if si > endi     # lwsp:  vbuf[ endi, si - endi ]
          @buf << vbuf[ si, i - si ].join('')
        end
      else
        if i == beg then
          #
          #  S
          #
          while (t = sbuf[i]) and t == :S do i += 1 end
          flush vbuf[ beg, i - beg ].join('')
        else
          #
          #  A
          #
          @buf << vbuf[ beg, i - beg ].join('')
        end
      end

      beg = i
    end

    flush('') unless @buf.empty?
  end

  
  MPREFIX = '=?iso-2022-jp?B?'
  MSUFFIX = '?='
  PRESIZE = MPREFIX.size + MSUFFIX.size
  
  def concat_e( arr )
    #
    # @buf must be empty, must be @lwsp exist
    #
    puts "concat_e: arr=#{arr.inspect}" if BENCODE_DEBUG
    
    # check if can concat lwsp

    if @width + @lwsp.size + @buf.size + encsize( arr[0].size ) <= @len then
      @f << @lwsp; @lwsp = ''
      @f << @buf;  @buf = ''
      @width += @lwsp.size + @buf.size
    else
      flush ''
    end

    tmp = ''
    arr.each do |i|
      if @width + encsize( tmp.size + i.size ) <= @len then
        tmp << i
      else
        e_cat tmp
        tmp = i
      end
    end
    e_cat tmp unless tmp.empty?

    if BENCODE_DEBUG then
      puts "concat_e: ret=#{@f.inspect}"
      puts "concat_e: buf=#{@buf.inspect}"
    end
  end

  def e_cat( str )
    puts "e_cat: start" if BENCODE_DEBUG

    str = [str].pack('m')
    str.chop!
    @buf << MPREFIX << str << MSUFFIX
    flush '', true
  end

  def encsize( len )
    amari = (if len % 3 == 0 then 0 else 1 end)
    (len / 3 + amari) * 4 + PRESIZE
  end


  def chunk_max( len )
    rest = len - PRESIZE
    rest = rest / 4 * 3 - 6
    rest -= rest % 2
    rest
  end

end


end    # module TMail
