#
# futils.rb
#
#   Copyright (c) 1999 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#


module FileUtils

  include FileTest

  @verbose = false
  @futils_output = $stderr

  def futilsmsg( msg )
    @futils_output ||= $stderr
    @futils_output.puts 'futils: ' + msg
  end


  def cd( dn )
    if iterator? then
      pre = Dir.pwd
      Dir.chdir dn
      yield
      Dir.chdir pre
    else
      Dir.chdir dn
    end
  end
  alias chdir cd

  def indir( dir, arg )
    if Array === arg then
      arg.collect{|fn| "#{dir}/#{fn}" }
    else
      "#{dir}/#{arg}"
    end
  end

  def isdir( dn )   # must not be array
    mkdir_p dn
    dn
  end

  def mustdir( dn )
    full = File.expand_path( dn )
    unless directory? full then
      raise ArgumentError, "#{dn} is not directory"
    end
    full
  end


  def expand( fn )
    File.expand_path fn
  end

  def dirname( fn )
    File.dirname fn
  end

  def basename( fn )
    File.basename fn
  end

  def fjoin( *args )
    File.join( *args )
  end
  alias sepjoin fjoin

  def is_newer?( newer, *older )
    return false unless exist? newer
    older.flatten.each do |i|
      return false if File.ctime(newer) < File.ctime(i)
    end

    true
  end

  alias uptodate? is_newer?

  def is_older?( older, *newer )
    return true unless exist? older
    newer.flatten.each do |i|
      return false if File.ctime(older) > File.ctime(i)
    end

    true
  end


  def i_mkdir( dn )
    return if directory? dn
    Dir.mkdir dn
  end

  def mkdir( *args )
    args.flatten!
    futilsmsg "mkdir #{args.join ' '}" if @verbose
    args.each do |i|
      i_mkdir File.expand_path(i)
    end
  end

  def i_mkdir_p( dn )
    return if directory? dn
    i_mkdir_p File.dirname dn
    Dir.mkdir dn unless File.basename( dn ) == ''
  end

  def mkdir_p( *args )
    args.flatten!
    futilsmsg "mkdir -p #{args.join ' '}" if @verbose
    args.each do |i|
      i_mkdir_p File.expand_path(i)
    end
  end
  alias mkpath mkdir_p


  def i_ln( old, new )
    dest = mkdestname( old, new )
    File.link old, dest
  end

  def ln( old, new )
    futilsmsg "ln #{old} #{new}" if @verbose
    i_ln old, new
  end

  def i_ln_s( old, new )
    dest = mkdestname( old, new )
    File.symlink old, new
  end

  def ln_s( old, new )
    futilsmsg "ln -s #{old} #{new}" if @verbose
    i_ln_s old, new
  end


  BSIZE = 1024 * 2  # 2KB

  def i_cp( from, to )
    dest = mkdestname( from, to )

    rf = File.open( from )     ; rf.binmode
    wf = File.open( dest, 'w' ); wf.binmode
    s = nil

    begin
      while s = rf.sysread( BSIZE * 16 ) do
        wf.syswrite s
      end
    rescue EOFError
    ensure
      rf.close
      wf.close
    end
  end

  def cp( *args )
    args.flatten!
    to = args.pop
    futilsmsg "cp #{args.join ' '} #{to}" if @verbose
    to = File.expand_path( to )
    args.each do |i|
      i_cp File.expand_path(i), to
    end
  end

  def cp_r( from, to )
    if directory? from then
      from = expand(from)
      do_cp_r dirname(from), basename(from), to
    else
      cp from, to
    end
  end

  def do_cp_r( base, abs, to )
    dirs = nil
    Dir.open( fjoin base, abs ) {|d| dirs = d.to_a }
    dirs.each do |fn|
      if directory? fn then
        next if /\A\.\.?\z/o === fn
        mkdir File.join(to, abs, fn)
        do_cp_r base, File.join(abs, fn), to
      else
        i_cp File.join(base, abs, fn), File.join(to, abs, fn)
      end
    end
  end


  def i_mv( from, to )
    dest = mkdestname( from, to )
    st = File.stat( from )
    if File::ALT_SEPARATOR then
      File.unlink dest
    end

    begin
      File.rename from, dest
    rescue
      if symlink? from then
        File.symlink File.readlink(from), dest
        File.unlink from
      else
        i_cp from, dest
        File.unlink from
        File.utime st.atime, st.mtime, dest
        begin
          File.chown st.uid, st.gid, dest
        rescue
        end
      end
    end
  end

  def mv( *args )
    args.flatten!
    to = args.pop
    futilsmsg "mv #{args.join ' '} #{to}" if @verbose
    to = File.expand_path( to )
    args.each do |i|
      i_mv File.expand_path(i), to
    end
  end


  def rm( *args )
    args.flatten!
    futilsmsg "rm #{args.join ' '}" if @verbose
    args.each do |i|
      File.unlink i
    end
  end

  def i_rm_f( fn )
    if exist? fn then
      unless file? fn then
        futilsmsg "'rm #{fn}' fail: not file"
        return
      end
      File.chmod 0777, fn
      File.unlink fn
    end
  end

  def rm_f( *args )
    args.flatten!
    futilsmsg "rm -f #{args.join ' '}" if @verbose
    args.each do |i|
      i_rm_f i
    end
  end

  def rm_rf( *args )
    args.flatten!
    args.each do |i|
      if directory? i then
        do_rmrf i
      else
        i_rm_f i
      end
    end
  end

  def do_rmrf( dn )
    foreach_fullpath( dn ) do |fn|
      if directory? fn then
        do_rmrf fn
      else
        i_rm_f fn
      end
    end
    Dir.rmdir dn
  end


  def i_cmp( filea, fileb )
    a = File.open( filea ); a.binmode
    b = File.open( fileb ); b.binmode

    stra = strb = ''
    begin
      while stra == strb do
        stra = a.read( BSIZE )
        strb = b.read( BSIZE )
        unless stra and strb then
          if stra.nil? and strb.nil? then
            return true
          end
        end
      end
    rescue EOFError
      ;
    ensure
      a.close
      b.close
    end

    false
  end

  def cmp( a, b )
    futilsmsg "cmp #{a} #{b}" if @verbose
    i_cmp a, b
  end

  alias identical? cmp


  def i_install( from, to, mode = nil )
    dest = mkdestname( from, to )
    unless exist? dest and i_cmp( from, dest ) then
      i_rm_f dest
      i_cp from, dest
      File.chmod mode, dest if mode
    end
  end

  def install( from, to, mode = nil )
    futilsmsg "install #{mode ? '%o ' % mode : ''}#{from} #{to}" if @verbose
    i_install from, to, mode
  end


  def foreach_fullpath( dn )
    d = Dir.open( dn )
    dirs = d.to_a
    d.close
    dirs.each do |fn|
      next if fn == '.' or fn == '..'
      yield dn + '/' + fn
    end
  end

  def chmod( mode, *args )
    args.flatten!
    futilsmsg "chmod #{mode ? '%o ' % mode : ''}#{args.join ' '}" if @verbose
    File.chmod mode, *args
  end

  Separator = File::ALT_SEPARATOR || File::SEPARATOR


  def mkdestname( from, to )
    if directory? to then
      (to[-1,1] == Separator ? to : to + Separator) + basename( from )
    else
      to
    end
  end

  def command( line )
    unless ret = system( line ) then
      raise ArgumentError, "'system #{line}' failed"
    end
    ret
  end


  module_function :cd, :chdir,
    :indir, :isdir, :mustdir,
    :expand, :dirname, :basename, :fjoin, :sepjoin,
    :is_newer?, :uptodate?, :is_older?,
    :mkdir, :mkdir_p, :mkpath,
    :ln, :ln_s, :cp, :cp_r, :mv, :rm, :rm_f, :rm_rf,
    :cmp, :identical?, :install,
    :foreach_fullpath, :chmod,
    :command

  # library private
  module_function :futilsmsg, :mkdestname, :i_mkdir, :i_mkdir_p, 
    :i_ln, :i_ln_s, :i_cp, :i_mv, :i_rm_f, :i_cmp, :i_install

end
