#
# futils.rb
#
#   Copyright (c) 1999 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#

require 'ftools'


module FileUtils

  include FileTest

  @verbose = false

  def chdir( dn )
    pre = Dir.pwd
    Dir.chdir dn
    yield
    Dir.chdir pre
  end

  def expand( fn )
    File.expand_path fn
  end

  def indir( dir, arg )
    if Array === arg then
      arg.collect{|fn| "#{dir}/#{fn}" }
    else
      "#{dir}/#{arg}"
    end
  end

  def dirname( fn )
    File.dirname fn
  end

  def basename( fn )
    File.basename fn
  end

  def fjoin( *args )
    File.join( *args )
  end
  alias sepjoin fjoin

  def is_newer?( newer, *older )
    return false unless exist? newer
    older.flatten.each do |i|
      return false if File.ctime(newer) < File.ctime(i)
    end
    return true
  end

  alias uptodate? is_newer?

  def is_older?( older, *newer )
    return true unless exist? older
    newer.flatten.each do |i|
      return false if File.ctime(older) > File.ctime(i)
    end
    return true
  end


  def install( from, to, mode = nil )
    File.install from, to, mode, @verbose
  end

  def mkdir( *dn )
    dn.flatten.each do |i|
      next if File.directory? i
      Dir.mkdir i
    end
  end

  def mkpath( *dn )
    File.mkpath *(dn.flatten)
  end

  def isdir( dn )   # must not be array
    mkpath dn
    dn
  end

  def mustdir( dn )
    full = expand( dn )
    unless directory? full then
      raise ArgumentError, "#{dn} is not directory"
    end
    full
  end


  BSIZE = 2048

  def ln( old, new )
    File.link old, new
  end

  def ln_s( old, new )
    File.symlink old, new
  end

  def cp( *args )
    args.flatten!
    to = args.pop
    args.each do |i|
      File.cp i, to, @verbose
    end
  end

  def cp_r( from, to )
    if directory? from then
      from = expand(from)
      do_cp_r dirname(from), basename(from), to
    else
      cp from, to
    end
  end

  def do_cp_r( base, abs, to )
    d = Dir.open( fjoin base, abs )
    dirs = d.to_a
    d.close
    dirs.each do |fn|
      if directory? fn then
        next if /\A\.\.?\z/o === fn
        mkdir fjoin(to, abs, fn)
        do_cp_r base, fjoin(abs, fn), to
      else
        cp fjoin(base, abs, fn), fjoin(to, abs, fn)
      end
    end
  end
  private :do_cp_r

  def mv( *args )
    args.flatten!
    to = args.pop
    args.each do |i|
      File.mv i, to, @verbose
    end
  end

  def rm( *fn )
    fn.flatten.each do |i|
      File.unlink i
      @stderr.puts "rm #{i}" if @verbose
    end
  end

  def rm_f( *fn )
    fn.flatten.each do |i|
      File.rm_f i, @verbose
    end
  end

  def rm_rf( *dn )
    begin
      dn.flatten.each do |i|
        if directory? i then
          do_rmrf i
        else
          File.rm_f i, @verbose
        end
      end
    rescue
      puts "error in rm_rf"
      $stderr.puts $!
    end
  end

  def do_rmrf( dn )
    foreach_fullpath( dn ) do |fn|
      if directory? fn then
        do_rmrf fn
      else
        File.rm_f fn, @verbose
      end
    end
    Dir.rmdir dn
  end
  private :do_rmrf

  def foreach_fullpath( dn )
    d = Dir.open( dn )
    dirs = d.to_a
    d.close
    dirs.each do |fn|
      next if fn == '.' or fn == '..'
      yield dn + '/' + fn
    end
  end

  def chmod( mod, *fn )
    fn.flatten!
    fn.push @verbose
    File.chmod mod, *fn
  end

  module_function :expand, :indir, :dirname, :basename,
    :fjoin, :sepjoin,
    :is_newer?, :uptodate?, :is_older?,
    :install, :mkdir, :mkpath, :isdir,
    :ln, :ln_s, :cp, :cp_r, :mv, :rm, :rm_f, :rm_rf,
    :foreach_fullpath, :chmod

end
