#
# field.rb
#
#     Copyright(c) 1998-1999 Minero Aoki
#     aamine@dp.u-netsurf.ne.jp
#

require 'delegate'

require 'amstd/extmod'

require 'tmail/bencode'
require 'tmail/mailp'



class HeaderField

  def initialize( fname, fbody )
    fname.must String
    fbody.must String

    @name = fname
    fbody.strip!
    @body = fbody

    @parsed = false
    @parsing = false
  end


  class << self

    alias hf_original_new new

    def new( fname, fbody )
      if self == HeaderField then
        temp = fname.downcase
        if /\Ax-/ === temp then
          cls = StringH
        else
          cls = STR2CLASS[temp] || UnknownH
        end

        cls.hf_original_new( fname, fbody )
      else
        hf_original_new( fname, fbody )
      end
    end


    def parse_on_rw( nm, tp, r, w )
      nm = nm.id2name

      if r then
        module_eval %^
          def #{nm}
            unless @parsed then
              parse
            end
            @#{nm}
          end
        ^
      end
      if w then
        module_eval %^
          def #{nm}=( arg )
            unless @parsed and not @parsing then
              parse
              @parsed = true
            end
            @#{nm} = arg
          end
        ^
      end
    end

    def parse_on_create
      module_eval %^
        def initialize( fname, fbody )
          super
          parse
        end
      ^
    end
  
  end


  def hash
    @name.hash
  end

  def eql?( oth )
    self.type === oth and @name == oth.name
  end

  def ==( oth )   # don't alias
    eql? oth
  end
    

  abstract :body

  def name
    @name.dup
  end

  def to_s( sep = "\n" )
    @name + ': ' + Bencode.encode( self.body, sep + ' ' )
  end

  def inspect( sep = nil )   # DON'T remove default (for TMail#inspect)
    @name + ': ' + self.body
  end

end



# HString  ---------------------------------------------


class StringH < HeaderField

  def initialize( fn, bo )
    super
    @deced = nil
  end

  def body
    unless @deced then
      @deced = Bencode.decode( @body )
    end
    @deced
  end

  def body=( str )
    str.must String
    @body = str
    @deced = nil
  end

  def eql?( oth )
    super and
    body == oth.body
  end

end


# struct type -----------------------------------------

class StructH < HeaderField

  def initialize( nm, bo )
    super
    @comments = []
  end

  def comments
    unless @parsed then
      parse
      @parsed = true
    end
    @comments
  end

  abstract :body

  def eql?( oth )
    super and
    @body == oth.body
  end


  private

  def init
    @comments.clear
  end

  def parse
    unless @parsing then
      @parsing = true
      init
      Mailp.parse( self, @body )
      @parsing = false
    end
  end

end


# unknown type ----------------------------------------

class UnknownH < StructH

  def initialize( fname, body )
    super( fname, body )
    @deced = nil
  end

  def body
    unless @deced then
      @deced = Bencode.decode( @body )
    end

    @deced
  end

  def body=( str )
    @deced = Bencode.decode( str )
  end

  def eql?( oth )
    super and
    @deced == oth.body
  end

end


# date type -------------------------------------------

class DateH < StructH

  parse_on_create

  property :date, Time, true, false

  def date=( arg )
    arg.must Time
    @date = arg.localtime
  end

  def body
    DateH.t2s( @date )
  end

  def eql?( oth )
    super and
    self.date == oth.date
  end

end


class << DateH

  def zone_s2i( str )
    temp = nil

    case str
    when /jst/io then temp = 9
    when /([\+\-])(\d\d?)(\d\d)/o then
      min  = $2.to_i * 60 + $3.to_i
      if $1 == '-' then
        ret = min * (-1)
      else
        ret = min
      end
    when /gmt/io then temp = 0
    when /utc/io then temp = 0
    when /ut/io  then temp = 0
    when /edt/io then temp = -4
    when /est/io then temp = -5
    when /cdt/io then temp = -5
    when /cst/io then temp = -6
    when /mdt/io then temp = -6
    when /mst/io then temp = -7
    when /pdt/io then temp = -7
    when /pst/io then temp = -8
    when /a/io   then temp = -1
    when /b/io   then temp = -2
    when /c/io   then temp = -3
    when /d/io   then temp = -4
    when /e/io   then temp = -5
    when /f/io   then temp = -6
    when /g/io   then temp = -7
    when /h/io   then temp = -8
    when /i/io   then temp = -9
    # J not use
    when /k/io   then temp = -10
    when /l/io   then temp = -11
    when /m/io   then temp = -12
    when /n/io   then temp = 1
    when /o/io   then temp = 2
    when /p/io   then temp = 3
    when /q/io   then temp = 4
    when /e/io   then temp = 5
    when /s/io   then temp = 6
    when /t/io   then temp = 7
    when /u/io   then temp = 8
    when /v/io   then temp = 9
    when /w/io   then temp = 10
    when /x/io   then temp = 11
    when /y/io   then temp = 12
    when /z/io   then temp = 0
    else
      raise ArgumentError, "wrong timezone format '#{str}'"
    end

    if temp then ret = temp * 60 end

    return ret
  end


  def t2s( tim )
    tim.must Time

    ret = tim.strftime( "%a, #{tim.mday} %b %Y %X " )

    # Time#gmtime changes self!!!
    tm = Time.at( tim.to_i )
    # ref [ruby-list:7928] --
    offset = tm.to_i - Time.mktime( *tm.gmtime.to_a[0, 6].reverse ).to_i
    ret << '%+.2d%.2d' % (offset / 60).divmod( 60 )

    return ret
  end

end


# return path type -------------------------------------

class RetpathH < StructH

  parse_on_create

  property :routes, Array,  true, false
  property :addr,   String, true, true

  def body
    str = '<'
    if @routes.size > 0 then
      str << @routes.collect{|i| '@' + i }.join(',') << ':'
    end
    str << @addr << '>'
  end

  def eql?( oth )
    super and
    self.addr == oth.addr and
    self.routes == oth.routes
  end


  private

  def init
    super
    @routes = []
  end

end


# address classes ---------------------------------------------


module MailAddr ; end


class Mbox

  include MailAddr

  def initialize( adr, phr = '', ro = nil )
    self.addr   = adr
    self.phrase = phr

    if ro then
      ro.must Array
      @routes = ro
    else
      @routes = []
    end
  end


  property :addr,   String, true, true
  property :routes, Array,  true, false
  property :phrase, String, true, true


  def eql?( oth )
    Mbox === othe and
    self.addr   == oth.addr   and
    self.routes == oth.routes and
    self.phrase == oth.phrase
  end
  alias == eql?


  def to_s
    str = ''
    str << Bencode.encode( @phrase ) << ' ' unless @phrase.empty?
    add_addr str
  end

  def inspect
    str = ''
    str << @phrase << ' ' unless @phrase.empty?
    add_addr str
  end
  
  def add_addr( str )
    br = str.empty? and @routes.empty?

    str << '<' unless br
    str << @routes.collect{|i| '@' + i }.join(',') << ':' unless @routes.empty?
    str << @addr
    str << '>' unless br

    str
  end
  private :add_addr

end   # class Mbox



class AddrGroup < DelegateClass( Array )

  def initialize( name, arg = nil )
    @name = (name.must String)

    if arg then
      super( arg.must(Array).dup )
    else
      super( [] )
    end
  end


  include Enumerable
  include MailAddr

  property :name, String, true, false

  
  def eql?( oth )
    super( oth ) and self.name == oth.name
  end

  def each_addr( &block )
    each do |i|
      case i
      when AddrGroup then i.each_addr( &block )
      when Mbox      then block.call( i.to_s )
      else
        bug! "wrong type #{i.type} in addr group"
      end
    end
  end
  
  def to_s
    @name + ':' << @items.join(', ') << ';'
  end

end   # class AddrGroup


# saddr type -------------------------------------------

class SaddrH < StructH
  
  parse_on_create

  property :addr, :Mbox, true, true

  def body
    @addr.inspect
  end

  def eql?( oth )
    super and
    self.addr == oth.addr
  end

end

class SmboxH < SaddrH
end


# maddr type -----------------------------------------

class MaddrH < StructH

  parse_on_create

  property :addrs, Array, true, false

  def body
    @addrs.collect{|a| a.inspect}.join(', ')
  end

  def eql?( oth )
    super and
    self.addrs == oth.addrs
  end


  private

  def init
    super
    @addrs = []
  end

end


class MmboxH < MaddrH
end


# ref type -------------------------------------------

class RefH < StructH

  parse_on_rw :refs, Array, true, false

  def body
    if @parsed then
      @refs.join(' ')
    else
      @body.dup
    end
  end

  def eql?( oth )
    super and
    self.refs == oth.refs
  end

  def each_msgid
    refs.each do |i|
      yield i if MsgidH.msgid? i
    end
  end

  def each_phrase
    refs.each do |i|
      yield i unless MsgidH.msgid? i
    end
  end


  private

  def init
    super
    @refs = []
  end

end


# key type -------------------------------------------

class KeyH < StructH

  parse_on_rw :keys, Array, true, false

  def body
    if @parsed then
      @keys.join(', ')
    else
      @body.dup
    end
  end

  def eql?( oth )
    super and
    self.keys == oth.keys
  end


  private

  def init
    super
    @keys = []
  end

end


# received type ---------------------------------------


class RecvH < StructH

  parse_on_rw :from,  [ :String, :NilClass ], true, true
  parse_on_rw :by,    [ :String, :NilClass ], true, true
  parse_on_rw :via,   [ :String, :NilClass ], true, true
  parse_on_rw :with,  [ :Array,  :NilClass ], true, false
  parse_on_rw :msgid, [ :String, :NilClass ], true, true
  parse_on_rw :ford,  [ :Mbox,   :NilClass ], true, true
  parse_on_rw :date,  [ :Time,   :NilClass ], true, true

  def body
    unless @parsed then
      return @body.dup
    end

    ret = ''

    ret << 'from ' << @from << ' ' if @from
    ret << 'by '   << @by   << ' ' if @by
    ret << 'via '  << @via  << ' ' if @via

    @with.each do |i| ret << 'with ' << i << ' ' end

    ret << 'id '   << @msgid << ' ' if @msgid
    ret << 'for '  << @ford.to_s    if @ford

    if @date then
      ret.strip!
      ret << '; '
      ret << DateH.t2s( @date )
    end

    return ret
  end

  def eql?( oth )
    super and
    self.from  == oth.from and
    self.by    == oth.by and
    self.via   == oth.via and
    self.with  == oth.with and
    self.msgid == oth.msgid and
    self.ford  == oth.ford and
    self.date  == oth.date
  end


  private

  def init
    super
    @with = []
  end

end


# message-id type  ----------------------------------------------------

class MsgidH < StructH

  def initialize( nm, str )
    super
    @msgid = str.strip
  end

  property :msgid, String, true, true
  

  MSGID = /<[^\@>]+\@[^>\@]+>/o

  def MsgidH.msgid?( str )
    str.must String
    MSGID === str
  end

  def body
    @msgid.dup
  end

  def eql?( oth )
    super and
    self.msgid == oth.msgid
  end

end



# encrypted type ------------------------------------------------------

class EncH < StructH

  parse_on_rw :encrypter,   :String,              true, true
  parse_on_rw :keyword,   [ :String, :NilClass ], true, true

  def body
    if @key then
      @encrypter + ', ' + @key
    else
      @encrypter.dup
    end
  end

  def eql?( oth )
    super and
    self.encrypter == oth.encrypter and
    self.keyword   == oth.keyword
  end

end



# version type -----------------------------------------

class VersionH < StructH

  parse_on_rw :major, :Integer, true, true
  parse_on_rw :minor, :Integer, true, true

  def body
    if @parsed then
      "#{@major}.#{@minor}"
    else
      @body.dup
    end
  end

  def eql?( oth )
    super and
    self.major == oth.major and
    self.minor == oth.minor
  end

end



# content type  ---------------------------------------


class CTypeH < StructH

  parse_on_create

  property :main,   :String, true, true
  property :sub,    :String, true, true
  property :params, :Hash,   true, false

  def []( key )
    params[key]
  end

  def body
    ret = "#{@main}/#{@sub}"
    params.each_pair do |k,v|
      ret << '; ' << k << '=' << v
    end

    return ret
  end
  
  def eql?( oth )
    super and
    self.main == oth.main and
    self.sub  == oth.sub  and
    self.params == oth.params
  end


  private

  def init
    super
    @params = {}
  end

end


# encoding type  ----------------------------


class CEncodingH < StructH

  parse_on_create

  property :encoding, :String, true, false

  def encoding=( enc )
    enc.must String
    @encoding = enc
  end

  def body
    @encoding.dup
  end
  
  def eql?( oth )
    super and
    self.encoding == oth.encoding
  end

end



# disposition type ----------------------------------

class CDispositionH < StructH

  parse_on_create

  property :disposition, :String, true, true
  property :params,      :Hash,   true, false

  def []( key )
    params[key]
  end

  def body
    ret = @disposition.dup
    @params.each_pair do |k,v|
      ret << '; ' << k << '=' << v
    end
    
    return ret
  end

  def eql?( oth )
    super and
    self.disposition == oth.disposition and
    self.params == oth.params
  end
  

  private

  def init
    super
    @params = {}
  end
    
end


# ---------------------------------------------------

class HeaderField   # backward definition

  STR2CLASS = {
    'date'                      => DateH,
    'resent-date'               => DateH,
    'received'                  => RecvH,
    'return-path'               => RetpathH,
    'sender'                    => SaddrH,
    'resent-sender'             => SaddrH,
    'to'                        => MaddrH,
    'cc'                        => MaddrH,
    'bcc'                       => MaddrH,
    'from'                      => MmboxH,
    'reply-to'                  => MaddrH,
    'resent-to'                 => MaddrH,
    'resent-cc'                 => MaddrH,
    'resent-bcc'                => MaddrH,
    'resent-from'               => MmboxH,
    'resent-reply-to'           => MaddrH,
    'message-id'                => MsgidH,
    'resent-message-id'         => MsgidH,
    'content-id'                => MsgidH,
    'in-reply-to'               => RefH,
    'references'                => RefH,
    'keywords'                  => KeyH,
    'encrypted'                 => EncH,
    'mime-version'              => VersionH,
    'content-type'              => CTypeH,
    'content-transfer-encoding' => CEncodingH,
    'content-disposition'       => CDispositionH,
    'subject'                   => StringH,
    'comments'                  => StringH,
    'content-description'       => StringH
  }

end
