#
# port.rb
#
#     Copyright(c) 1999 Minero Aoki
#     aamine@dp.u-netsurf.ne.jp
#

require 'ftools'


class Port
end


class Stream

  CR = "\r"[0]
  LF = "\n"[0]

  def interrupted?
    @intr
  end

  attr :term, true

end



class StringPort < Port

  def initialize( str )
    super()
    @str = str
  end

  def ropen
    p = StringIStream.new( @str )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def wopen
    raise NameError,
      "StringPort\#wopen: operation not permitted"

    p = StringOStream.new( @str )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def aopen
    raise NameError,
      "StringPort\#aopen: operation not permitted"

    p = StringOStream.new( @str )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def rm
    @str.replace ''
    @str.freeze
  end

  def cp( port )
    if StringPort === port then
      port.str.replace @str
    else
      port.wopen do |s|
        s.write @str
      end
    end
  end

  def mv( port )
    cp port
    port.rm
  end

  def size
    @str.size
  end

  def eql?( other )
    StringPort === other and @str == other.str
  end
  alias == eql?

  def hash
    @str.hash
  end


  protected
  
  attr :str

end


class StringIStream < Stream

  def initialize( str )
    @str = str
    @intr = false
    @pos = 0
    @term = nil
  end

  def interrupt
    @intr = true
  end

  def restart
    @intr = false
  end

  def close
    @intr = true
    @str = nil
  end

  def to_s
    "<StringIStream:intr=#{@intr},@str=#{@str[0,30].inspect}>"
  end

  def rewind
    @pos = 0
  end

  def each( &block )
    setterm unless @term
    @str.each( @term ){|l| yield l }
  end

  def gets
    setterm unless @term
    if n = @str.index( @term, @pos )
      ret = @str[ @pos, n + @term.size - @pos ]
      @pos = n + @term.size
      if @pos == @str.size then @pos += 1 end
    else
      ret = @str[ @pos, @str.size - @pos ]
      @pos = @str.size + 1
    end

    return ret
  end

  def getc
    ret = @str[ @pos ]
    @pos += 1
    return ret
  end

  def readall
    ret = @str[ @pos, @str.size - @pos ]
    @pos = @str.size + 1
    return ret
  end

  def read( num )
    ret = @str[ @pos, num ]
    @pos += num
    if @pos == @str.size then @pos += 1 end
    return ret
  end


  private

  def setterm
    if /\n|\r\n|\r/o === @str then
      @term = $&
    else
      @term = "\n"
    end
  end

end


class StringOStream < Stream

  def initialize( str )
    @str = str
    @term = "\n"
    @intr = false
  end

  def interrupt
    @intr = true
  end

  def restart
    @intr = false
  end

  def close
    @intr = true
    @str = nil
  end

  def to_s
    "<StringOStream:intr=#{@intr},@str=#{@str[0,30].inspect}>"
  end

  def rewind
    @str.replace ''
  end

  def puts( str = nil )
    @str << str << if str
    @str << @term
  end

  def putc( ch )
    @str << ch.chr
  end

  def write( str )
    @str << str
  end

  def <<( str )
    @str << str
    self
  end

end



class FilePort < Port

  def initialize( fname )
    super()
    @filename = File.expand_path( fname.must String )
  end

  def ropen
    p = FileIStream.new( @filename )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def wopen
    p = FileOStream.new( @filename )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def aopen
    p = FileOStream.new( @filename, 'a' )
    if iterator? then
      yield p
      p.close
    else
      return p
    end
  end

  def to_s
    "<FilePort:locate=#{@filename}>"
  end

  def dup( mod = 'r' )
    ret = self.type.new( @filename, mod )
    ret.term = @term
    return ret
  end

  def rm
    File.unlink( @filename ) if File.file?( @filename )
  end

  def mv( port )
    if FilePort === port then
      File.mv @filename, port.filename
    else
      cp port
    end
    rm
  end

  def cp( port )
    ropen do |rp|
      port.wopen do |wp|
        while b = rp.read( 2048 ) do
          wp.write b
        end
      end
    end
  end

  def size
    File.size @filename
  end

  def eql?( other )
    FilePort === other and @filename == other.filename
  end
  alias == eql?

  def hash
    @filename.hash
  end


  protected

  attr :filename

end


class FileIStream < Stream

  def initialize( fname )
    @filename = (fname.must String)
    @f = File.open( fname, 'r' )
    @term = nil
  end

  def interrupt
    @intr = true
    @preserve = @f.tell
    @f.close
  end

  def restart
    @f = File.open( @filename, 'r' )
    @f.seek @preserve, 0
    @intr = false
  end

  def close
    @intr = true
    @f.close
    @filename = nil
  end

  def to_s
    "<FilePort:close=#{@intr},fname=#{@filename}>"
  end

  def each
    setterm unless @term
    @f.each( @term ){|l| yield l }
  end

  def gets
    setterm unless @term
    @f.gets @term
  end

  def getc
    @f.getc
  end

  def readall
    @f.read
  end

  def read( num )
    setterm unless @term
    @f.read num
  end


  private

  def setterm
    @term = "\n"
    pre = @f.tell
    while ch = @f.getc do
      if ch == CR then
        if @f.getc == LF then
          @term = "\r\n" ; break
        else
          @term = "\r" ; break
        end
      elsif ch == LF then
        @term = "\n" ; break
      end
    end
    @f.seek pre, 0
  end

end


class FileOStream < Stream
  
  def initialize( fname, mode = 'w' )
    fname.must String
    @filename = fname
    @intr = false
    @f = File.open( fname, mode )
    @term = "\n"
  end

  def interrupt
    @intr = true
    @preserve = @f.tell
    @f.close
  end

  def restart
    @f = File.open( @filename, 'w' )
    @f.seek @preserve, 0
    @intr = false
  end

  def close
    @f.close
    @intr = true
    @filename = nil
  end

  def puts( str = nil )
    @term = "\n" unless @term
    @f.write str if str
    @f.write @term
  end

  def putc( ch )
    @f.putc ch
  end

  def write( str )
    @f.write str
  end

  def <<( str )
    @f.write str
    self
  end

  def rewind
    @f.rewind
  end

end


class Port

  ARG_TO_CLASS = {
    String => StringPort,
    File   => FilePort
  }

  class << self

    alias port_original_new new

    def new( arg )
      if self == Port then
        if Port === arg then
          return arg
        end

        arr = ARG_TO_CLASS.find{|arg_type,port_type|
          if arg_type === arg then true
          else                     false
          end
        }
        unless arr then
          raise TypeError, "can't handle type #{arg.type}"
        end

        arr[1].port_original_new( arg )
      else
        self.port_original_new( arg )
      end
    end

  end

end
