/* vi:set ts=4 sw=4:
--------------------------------------------------------------------

  strscan.c version 0.2.0

    Copyright(c) 1999 Minero Aoki
    aamine@dp.u-netsurf.ne.jp

--------------------------------------------------------------------
*/


#include <stdio.h>
#ifdef __STDC__
#  include <stdlib.h>
#endif

#include "ruby.h"
#include "re.h"


struct strscanner
{
    char *pbeg;
    char *ptr;
    char *pend;
    char *preserve;
    int dupped;
};

#define GET_SCANNER(obj,dat) Data_Get_Struct(obj, struct strscanner, dat)
#define SCAN_FINISHED(dat) ((dat)->ptr >= (dat)->pend)

static VALUE rb_cStrScan;
static VALUE rb_eScanError;


/* ------------------------------------------------------------- */


static void
strscan_free(dat)
    struct strscanner *dat;
{
    if (dat->dupped)
        free(dat->pbeg);
    free(dat);
}


static VALUE
strscan_reset(self, str, dupp)
    VALUE self, str, dupp;
{
    struct strscanner *dat;
    char *ptr;
    int len;

    Check_Type(str, T_STRING);
    GET_SCANNER(self, dat);
    ptr = RSTRING(str)->ptr;
    len = RSTRING(str)->len;

    if (RTEST(dupp)) {
        dat->dupped = 1 ;
        dat->ptr = ALLOC_N(char, len) ;
        memcpy(dat->ptr, ptr, sizeof(char) * (len));
    } else {
        dat->dupped = 0;
        dat->ptr = ptr;
    }
    dat->pbeg = dat->ptr;
    dat->pend = dat->ptr + len;
    dat->preserve = 0;

    return Qnil;
}


static VALUE
strscan_new(argc, argv, cls)
    int argc;
    VALUE *argv, cls;
{
    VALUE obj;
    VALUE str;
    VALUE dupp;
    struct strscanner *dat;

    if (rb_scan_args(argc, argv, "11", &str, &dupp) == 1)
        dupp = Qtrue;
    obj = Data_Make_Struct(cls, struct strscanner, 0, strscan_free, dat);
    strscan_reset(obj, str, dupp);
    rb_obj_call_init(obj, argc, argv);

    return obj;
}


static VALUE
strscan_do_scan(self, regex, ptrflag, strflag)
    VALUE self, regex;
    int ptrflag, strflag;
{
    struct strscanner *dat;
    struct re_registers regi ;
    size_t len ;
    int ret ;

    Check_Type(regex, T_REGEXP);
    GET_SCANNER(self, dat);

    if (SCAN_FINISHED(dat)) return Qnil;
        
    MEMZERO(&regi, struct re_registers, 1);  /* this is important !!! */

    ret = re_match(RREGEXP(regex)->ptr,
                   dat->ptr,
                   dat->pend - dat->ptr,
                   0,
                   &regi);
    len = regi.end[0] ;
    ruby_re_free_registers(&regi);
        
    if (ret == -2) {
        rb_raise(rb_eScanError, "regexp buffer overflow");
    } else if ( ret < 0 ) {
        return Qnil ;
    } else {
        if (ptrflag) {
            dat->preserve = dat->ptr ;
            dat->ptr += len;
            if (strflag)
                return rb_str_new(dat->preserve, len);
        } else {
            if (strflag)
                return rb_str_new(dat->ptr, len);
        }
        return INT2FIX(len);
    }

    return Qnil; /* not reach */
}


static VALUE
strscan_fullscan(self, reg, strflag, ptrflag)
    VALUE self, reg, strflag, ptrflag;
{
    return strscan_do_scan(self, reg, RTEST(ptrflag), RTEST(strflag));
}


static VALUE
strscan_scan(self, reg)
    VALUE self, reg;
{
    return strscan_do_scan(self, reg, 1, 1);
}


static VALUE
strscan_match_p(self, reg)
    VALUE self, reg;
{
    return strscan_do_scan(self, reg, 0, 0);
}


static VALUE
strscan_skip(self, reg)
    VALUE self, reg;
{
    return strscan_do_scan(self, reg, 1, 0);
}


static VALUE
strscan_getch(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    if (SCAN_FINISHED(dat))
        return Qnil ;
    
    dat->preserve = dat->ptr;
    dat->ptr++;
    return rb_str_new(dat->preserve, 1);
}


static VALUE
strscan_rest(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    return rb_str_new(dat->ptr, dat->pend - dat->ptr);
}


static VALUE
strscan_rest_p(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    if (SCAN_FINISHED(dat))
        return Qfalse;
    else
        return Qtrue;
}


static VALUE
strscan_restsize(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    return INT2FIX(dat->pend - dat->ptr);
}


#define SCANNED_CHK(dat) {\
    if (dat->preserve == 0) {                                   \
        rb_raise(rb_eScanError, "not scanned: cannot unscan");  \
        return Qnil ;                                           \
    }                                                           \
}

static VALUE
strscan_unscan(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    SCANNED_CHK(dat);
    dat->ptr = dat->preserve;
    dat->preserve = 0;
    return Qtrue;
}


static VALUE
strscan_matched(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    SCANNED_CHK(dat);
    return rb_str_new(dat->preserve, dat->ptr - dat->preserve);
}


static VALUE
strscan_matchedsize(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    if (dat->preserve == 0)
        return Qnil;
    else
        return INT2FIX(dat->ptr - dat->preserve);
}


static VALUE
strscan_original(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    return rb_str_new(dat->pbeg, dat->pend - dat->pbeg);
}


static VALUE
strscan_clear(self)
    VALUE self;
{
    struct strscanner *dat;

    GET_SCANNER(self, dat);
    dat->ptr = dat->pend;
    dat->preserve = 0;
    return self;
}


static VALUE
strscan_peep(self, vlen)
    VALUE self, vlen;
{
    struct strscanner*dat;
    int temp;
    int len;

    GET_SCANNER(self, dat);

    len = NUM2INT(vlen);
    temp = dat->pend - dat->ptr;
    if (temp < len)
        len = temp;
    
    return rb_str_new(dat->ptr, len);
}



void
Init_strscan()
{
    rb_eScanError = rb_define_class("ScanError", rb_eStandardError);
    rb_cStrScan = rb_define_class("StrScanner", rb_cObject);

    rb_define_singleton_method(rb_cStrScan, "new", strscan_new, -1);
    rb_define_method(rb_cStrScan, "reset",       strscan_reset,       2);
    rb_define_method(rb_cStrScan, "fullscan",    strscan_fullscan,    3);
    rb_define_method(rb_cStrScan, "scan",        strscan_scan,        1);
    rb_define_method(rb_cStrScan, "skip",        strscan_skip,        1);
    rb_define_method(rb_cStrScan, "match?",      strscan_match_p,     1);
    rb_define_method(rb_cStrScan, "getch",       strscan_getch,       0);
    rb_define_method(rb_cStrScan, "rest",        strscan_rest,        0);
    rb_define_method(rb_cStrScan, "rest?",       strscan_rest_p,      0);
    rb_define_method(rb_cStrScan, "restsize",    strscan_restsize,    0);
    rb_define_method(rb_cStrScan, "unscan",      strscan_unscan,      0);
    rb_define_method(rb_cStrScan, "matched",     strscan_matched,     0);
    rb_define_method(rb_cStrScan, "matchedsize", strscan_matchedsize, 0);
    rb_define_method(rb_cStrScan, "original",    strscan_original,    0);
    rb_define_method(rb_cStrScan, "clear",       strscan_clear,       0);
    rb_define_method(rb_cStrScan, "peep",        strscan_peep,        1);
}
