#
# bencode.rb
#
#     Copyright(c) 1999 Minero Aoki
#     aamine@dp.u-netsurf.ne.jp
#
# usage:
#
#   encoded = Bencode.encode( plain_text )
#   decoded = Bencode.decode( b_encoded_word )
#

require 'must'
require 'bug'
require 'recycle'
require 'nkf'

$BENCODE_DEBUG = false

class Bencode

  include Recyclable


  def initialize( sepstr = "\n ", mlen = 72 )
    @varr = []
    @sarr = []
    @sbuf = []
    @vbuf = []
    @stmp = []
    @vtmp = []

    reset( sepstr, mlen )
  end


  def Bencode.encode( str, sepstr = "\n ", mlen = 72 )
    ret = nil
    use_instance( sepstr, mlen ) do |inst|
      ret = inst.encode( str )
    end
    return ret
  end


  Encoded   = /\=\?iso-2022-jp\?b\?[^\?]+\?\=/io
  SpEncoded = /\A\s+(\=\?iso-2022-jp\?b\?[^\?]+\?\=)/io

  def Bencode.decode( str, outcode = '-e' )
    ret = NKF.nkf( outcode + 'm', str )
=begin
    ret = ''
    while str.size > 0 do
      if Encoded === str
        str = $'
        ret << $`
        ret << NKF.nkf( outcode, NKF.nkf( '-m', $& ) )
        while SpEncoded === str do
          ret << NKF.nkf( outcode, NKF.nkf( '-m', $1 ) )
          str = $'
        end
      else
        ret << str
        break
      end
    end
    if outcode == '-j' then
      ret.gsub!( /\e\(\$\e\(B/o, '' )
    end
=end

    return ret
  end


  def reset( sepstr = "\n ", mlen = 74 )
    @sep = sepstr
    @len = mlen - sepstr.size
    @chlen = chank_max( @len )

    @ret = ''
    @buf = ''
    @sbuf.clear
    @vbuf.clear
    @stmp.clear
    @vtmp.clear

    @firstflush = true
  end


  def decode( str, code = Kconv::EUC )
    Bencode.decode( str, code )
  end


  def encode( str )
    scan( NKF.nkf( '-j', str ) )
    split_word
    make_string

    return @ret
  end


  def add_string( str )
    scan( NKF.nkf( '-j', str ) )
    return self
  end
  alias << add_string
    

  def add_ascii( str )
    @sbuf.push str
    @vbuf.push :A
    return self
  end


  def add_jis( str )
    @sbuf.push :J
    @vbuf.push str
    return self
  end


  def add_lwsp( str )
    @sbuf.push :S
    @vbuf.push str
    return self
  end


  def add_phrase( str )
    scan( str, true )
    return self
  end


  def result
    split_word
    make_string

    return (@ret or '')
  end


  private


  def scan( str, force = false )
    
    while str.size > 0 do
      case str
      when /\A[^\e\t\r\n ]+/o
        if force then @sbuf.push :J
        else          @sbuf.push :A
        end
        @vbuf.push $&

      when /\A(\e..)([^\e\t\r\n ]+)(\e\(B)?/o
        pre  = $1
        temp = $2
        post = $3

        while temp.size > @chlen do
          @sbuf.push :J
          @vbuf.push( pre + temp[ 0, @chlen ] + post )
          temp = temp[ @chlen, temp.size - @chlen ]
        end
        @sbuf.push :J
        @vbuf.push( pre + temp + post )

      when /\A[\t\r\n ]+/o
        @sbuf.push :S
        @vbuf.push $&

      else
        bug! "Bencode#scan, not match"
      end
      str = $'
    end
  end


  def split_word
    #
    # @sbuf = [ (J|A)(J|A|S)*(J|A) ) ]
    #
    #   A: ascii only, no LWSP
    #   J: jis string, no LWSP/ascii
    #   S: LWSP
    #
    # reduce /(J|A)+/ && /J/ to 'W', then @stmp = [ (W|A)(S(W|A))* ]
    #
    if $BENCODE_DEBUG then
      puts "\n\nenter split_word ------------\n\n"
      puts @sbuf.collect{|i| i.id2name }.inspect
      puts @vbuf.inspect
    end

    idx = 0
    jflag = false
    lflag = false

    while idx < @sbuf.size do

      sidx = idx
      while (item = @sbuf[idx]) do
        case item
        when :J
          jflag = true
        when :A
          ;
        when :S
          lflag = true
          break
        else
          bug! "in split.case: not match"
        end

        idx += 1
      end


      ## push word ##

      #if sidx > idx then
        if jflag then
          jflag = false
          @stmp.push :W
          @vtmp.push @vbuf[ sidx, idx - sidx ]
        else
          @stmp.push :A
          @vtmp.push @vbuf[ sidx ]
        end
      #end


      ## push lwsp ##

      if lflag then
        lflag = false
        @stmp.push :S
        @vtmp.push @vbuf[ idx ]
      end

      idx += 1
    end
  end

  
  def make_string
    #
    # A: ascii only (no LWSP)
    # S: LWSP
    # W: string include jis (no LWSP)
    # E: encoding piece
    #
    #   sarr = [ (W|A)(S(W|A)* ]
    #
    # reduce /W(SW)+/ to 'E', and
    # reduce /A(SA)+/ to 'D', then sarr = [ (E|A)(S(E|A))* ]
    #
    if $BENCODE_DEBUG then
      puts "\n\nenter make_string -------------\n\n"
      puts @stmp.collect{|i| i.id2name }.inspect
      puts @vtmp.inspect
    end

    idx = 1
    @stmp.unshift :S
    @vtmp.unshift ''

    while (sim = @stmp[idx]) do
      
      case sim
      when :A
        ascii_cat( @vtmp[idx - 1], @vtmp[idx] )
      when :W
        sidx = idx ; idx += 2
        while @stmp[idx] == :W do
          idx += 2
        end

        jis_cat( @vtmp[sidx - 1], @vtmp[sidx, idx - sidx - 1].flatten! )
        idx -= 2
      else
        bug! 'in make_string.while, not match'
      end

      idx += 2
    end

    if @buf.size > 0 then
      flush_buf(nil)
    end
  end


  def flush_buf( init )
    unless @firstflush then
      @ret << @sep
    else
      @firstflush = false
    end
    @ret << @buf
    @buf = init
  end

  
  def ascii_cat( sp, str )
    if $BENCODE_DEBUG then
      puts "ascii_cat: type=#{str.type}, str=#{str.inspect}"
    end

    if @buf.size + sp.size + str.size <= @len then
      @buf << sp << str
    else
      flush_buf( str )
    end

    if $BENCODE_DEBUG then
      puts "ascii_cat: ret=#{@ret.inspect}"
      puts "ascii_cat: buf=#{@buf.inspect}"
    end
  end
    
  
  MPREFIX = '=?iso-2022-jp?B?'
  MSUFFIX = '?='
  PRESIZE = MPREFIX.size + MSUFFIX.size
  
  def jis_cat( sp, arr )
    if $BENCODE_DEBUG then
      puts "jis_cat: type=#{arr.type}, arr=#{arr.inspect}"
      puts "jis_cat: sp.type=#{sp.type}, sp=#{sp}"
    end
    
    # check if can concat lwsp

    if @buf.size + sp.size + encsize( arr[0].size ) <= @len then
      @buf << sp
    else
      flush_buf( '' )
    end

    temp = ''
    arr.each do |i|
      if @buf.size + encsize( temp.size + i.size ) <= @len then
        temp << i
      else
        catjis( temp )
        flush_buf( '' )
        temp = i
      end
    end
    if temp.size > 0 then
      catjis( temp )
    end

    if $BENCODE_DEBUG then
      puts "ascii_cat: ret=#{@ret.inspect}"
      puts "ascii_cat: buf=#{@buf.inspect}"
    end
  end


  def catjis( str )
    puts "catjis: start" if $BENCODE_DEBUG

    @buf << MPREFIX << [str].pack('m')
    @buf.chop!
    @buf << MSUFFIX
  end


  def encsize( len )
    amari = (if len % 3 == 0 then 0 else 1 end)
    return( (len / 3 + amari) * 4 + PRESIZE )
  end

  def chank_max( len )
    rest = len - PRESIZE
    rest = rest / 4 * 3 - 6
    if rest % 2 != 0 then rest -= 1 end
    
    return rest
  end

end
