#
# facade.rb
#
#   Copyright (c) 1998-2002 Minero Aoki <aamine@loveruby.net>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'tmail/utils'


module TMail

  class Mail

    def header_string( name, default = nil )
      h = @header[name.downcase] or return default
      h.to_s
    end

    ###
    ### attributes
    ###

    include TextUtils

    def set_string_array_attr( key, strs )
      strs.flatten!
      if strs.empty? then
        @header.delete key.downcase
      else
        store key, strs.join(', ')
      end
      strs
    end
    private :set_string_array_attr

    def set_string_attr( key, str )
      if str then
        store key, str
      else
        @header.delete key.downcase
      end
      str
    end
    private :set_string_attr

    def set_addrfield( name, arg )
      if arg
        h = HeaderField.internal_new(name, @config)
        h.addrs.replace [arg].flatten
        @header[name] = h
      else
        @header.delete name
      end
      arg
    end
    private :set_addrfield


    #
    # date time
    #

    def date( default = nil )
      if header = @header['date'] then
        header.date
      else
        default
      end
    end

    def date=( time )
      if time then
        store 'Date', time2str(time)
      else
        @header.delete 'date'
      end
      time
    end

    def strftime( fmt, default = nil )
      if t = date then
        t.strftime(fmt)
      else
        default
      end
    end


    #
    # destination
    #

    def to_addrs( default = nil )
      if header = @header['to'] then
        header.addrs
      else
        default
      end
    end

    def cc_addrs( default = nil )
      if header = @header['cc'] then
        header.addrs
      else
        default
      end
    end

    def bcc_addrs( default = nil )
      if header = @header['bcc'] then
        header.addrs
      else
        default
      end
    end

    def to_addrs=( arg )
      set_addrfield 'to', arg
    end

    def cc_addrs=( arg )
      set_addrfield 'cc', arg
    end

    def bcc_addrs=( arg )
      set_addrfield 'bcc', arg
    end

    def to( default = nil )
      arr = to_addrs(nil) or return default
      arr.map {|a| a.spec }
    end

    def cc( default = nil )
      arr = cc_addrs(nil) or return default
      arr.map {|a| a.spec }
    end

    def bcc( default = nil )
      arr = bcc_addrs(nil) or return default
      arr.map {|a| a.spec }
    end

    def to=( *strs )
      set_string_array_attr 'To', strs
    end

    def cc=( *strs )
      set_string_array_attr 'Cc', strs
    end

    def bcc=( *strs )
      set_string_array_attr 'Bcc', strs
    end


    #
    # originator
    #

    def from_addrs( default = nil )
      if header = @header['from'] then
        header.addrs
      else
        default
      end
    end

    def from_addrs=( arg )
      set_addrfield 'from', arg
    end

    def from( default = nil )
      arr = from_addrs(nil) or return default
      arr.map {|a| a.spec }
    end

    def from=( *strs )
      set_string_array_attr 'From', strs
    end

    def friendly_from( default = nil )
      a, = from_addrs(nil)
      a or return default
      a.phrase || a.spec
    end


    def reply_to_addrs( default = nil )
      if header = @header['reply-to'] then
        header.addrs
      else
        default
      end
    end

    def reply_to_addrs=( arg )
      set_addrfield 'reply-to', arg
    end

    def reply_to( default = nil )
      arr = reply_to_addrs(nil) or return default
      arr.map {|a| a.spec }
    end

    def reply_to=( *strs )
      set_string_array_attr 'Reply-To', strs
    end


    def sender_addr( default = nil )
      f = @header['sender'] or return default
      f.addr                or return default
      f.addr
    end

    def sender_addr=( addr )
      if addr then
        h = HeaderField.internal_new('sender', @config)
        h.addr = addr
        @header['sender'] = h
      else
        @header.delete 'sender'
      end
      addr
    end

    def sender( default )
      f = @header['sender'] or return default
      a = f.addr            or return default
      a.spec
    end

    def sender=( str )
      set_string_attr 'Sender', str
    end


    #
    # subject
    #

    def subject( default = nil )
      if header = @header['subject'] then
        header.body
      else
        default
      end
    end

    def subject=( str )
      set_string_attr 'Subject', str
    end


    #
    # identity & threading
    #

    def message_id( default = nil )
      if header = @header['message-id'] then
        header.id
      else
        default
      end
    end

    def message_id=( str )
      set_string_attr 'Message-Id', str
    end

    def in_reply_to( default = nil )
      if header = @header['in-reply-to'] then
        header.ids
      else
        default
      end
    end

    def in_reply_to=( *idstrs )
      set_string_array_attr 'In-Reply-To', idstrs
    end

    def references( default = nil )
      if header = @header['references'] then
        header.refs
      else
        default
      end
    end

    def references=( *strs )
      set_string_array_attr 'References', strs
    end


    #
    # MIME headers
    #

    def mime_version( default = nil )
      if header = @header['mime-version'] then
        header.version
      else
        default
      end
    end

    def mime_version=( m, opt = nil )
      if opt then
        if f = @header['mime-version'] then
          f.major = m
          f.minor = opt
        else
          store 'Mime-Version', "#{m}.#{opt}"
        end
      else
        store 'Mime-Version', m
      end
      m
    end


    def content_type( default = nil )
      if header = @header['content-type'] then
        header.content_type
      else
        default
      end
    end

    def main_type( default = nil )
      if header = @header['content-type'] then
        header.main_type
      else
        default
      end
    end

    def sub_type( default = nil )
      if header = @header['content-type'] then
        header.sub_type
      else
        default
      end
    end

    def set_content_type( str, sub = nil, param = nil )
      if sub then
        main, sub = str, sub
      else
        main, sub = str.split('/', 2)
      end
      if f = @header['content-type'] then
        f.main_type = main
        f.sub_type  = sub
      else
        store 'Content-Type', "#{main}/#{sub}"
      end
      @header['content-type'].params.replace param if param

      str
    end

    alias content_type= set_content_type

    EMPTY_HASH = {}

    def type_params( default = EMPTY_HASH )
      if h = @header['content-type']
        h.params
      else
        default
      end
    end
    
    def type_param( name, default = nil )
      if h = @header['content-type']
        h.params[name] or default
      else
        default
      end
    end

    def charset( default = nil )
      if h = @header['content-type']
        h['charset'] or default
      else
        default
      end
    end

    def charset=( str )
      if str then
        if f = @header[ 'content-type' ] then
          f['charset'] = str
        else
          store 'Content-Type', "text/plain; charset=#{str}"
        end
      end
      str
    end


    def transfer_encoding( default = nil )
      if header = @header['content-transfer-encoding'] then
        header.encoding
      else
        default
      end
    end

    def transfer_encoding=( str )
      set_string_attr 'Content-Transfer-Encoding', str
    end

    alias encoding                   transfer_encoding
    alias encoding=                  transfer_encoding=
    alias content_transfer_encoding  transfer_encoding
    alias content_transfer_encoding= transfer_encoding=


    def disposition( default = nil )
      if header = @header['content-disposition'] then
        header.disposition
      else
        default
      end
    end

    def set_content_disposition( str, params = nil )
      if f = @header['content-disposition'] then
        f.disposition = str
      else
        f = store( 'Content-Disposition', str )
      end
      f.params.replace params if params
    end

    alias content_disposition  disposition
    alias content_disposition= set_content_disposition
    alias disposition=         set_content_disposition


    ###
    ### utils
    ###

    def create_reply
      mail = TMail::Mail.parse('')
      mail.subject = 'Re: ' + subject('').sub( /\A(?:\[[^\]]+])?(?:\s*Re:)*\s*/i, '' )
      mail.to_addrs = reply_addresses([])
      mail.in_reply_to = [message_id(nil)].compact
      mail.references = references([]) + [message_id(nil)].compact
      mail.mime_version = '1.0'
      mail
    end


    def base64_encode
      store 'Content-Transfer-Encoding', 'Base64'
      self.body = Base64.folding_encode(self.body)
    end

    def base64_decode
      if /base64/i === self.transfer_encoding('')
        store 'Content-Transfer-Encoding', '8bit'
        self.body = Base64.decode(self.body, @config.strict_base64decode?)
      end
    end


    def destinations( default = nil )
      ret = []
      %w( to cc bcc ).each do |nm|
        if head = @header[nm] then
          head.addrs.each {|i| ret.push i.address }
        end
      end
      ret.empty? ? default : ret
    end

    def each_destination( &block )
      destinations([]).each do |i|
        if Address === i then
          yield i
        else
          i.each( &block )
        end
      end
    end

    alias each_dest each_destination


    def reply_addresses( default = nil )
      reply_to_addrs(nil) or from_addrs(nil) or default
    end

    def error_reply_addresses( default = nil )
      if s = sender(nil) then
        [s]
      else
        from_addrs(default)
      end
    end


    def multipart?
      main_type('').downcase == 'multipart'
    end

  end   # class Mail

end   # module TMail
