
require 'runit/testcase'
require 'tmail/header'
require 'extctrl'


class UnstructuredHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Subject SUBJECT sUbJeCt
        X-My-Header ).each do |name|
      h = TMail::HeaderField.new(name, 'This is test header.')
      assert_instance_of TMail::UnstructuredHeader, h,
                         'Header.new: name=' + name.dump
    end
  end

  def test_to_s
    # I must write more and more test.
    [
      'This is test header.',
      # "This is \r\n\ttest header"
      # "JAPANESE STRING"
      ''
    ]\
    .each do |str|
      h = TMail::HeaderField.new('Subject', str)
      assert_equal str, h.decoded
      assert_equal str, h.to_s
    end
  end

end


class DateTimeHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Date Resent-Date ).each do |name|
      h = TMail::HeaderField.new(name, 'Tue, 4 Dec 2001 10:49:32 +0900')
      assert_instance_of TMail::DateTimeHeader, h, name
    end
  end

  def test_date
    h = TMail::HeaderField.new('Date', 'Tue, 4 Dec 2001 10:49:32 +0900')
    assert_instance_of Time, h.date
    assert_equal false, h.date.gmt?
    assert_equal Time.local(2001,12,4, 10,49,32), h.date
  end

  def test_empty__illegal?
    [ [false, 'Tue,  4 Dec 2001 10:49:32 +0900'],
      [false, 'Sat, 15 Dec 2001 12:51:38 +0900'],
      [true, 'Sat, 15 Dec 2001 12:51:38'],
      [true, 'Sat, 15 Dec 2001 12:51'],
      [true, 'Sat,'] ].each do |wrong, str|

      h = TMail::HeaderField.new('Date', str)
      assert_equal wrong, h.empty?, str
      assert_equal wrong, h.illegal?, str
    end
  end

  def test_to_s
    h = TMail::HeaderField.new('Date', 'Tue, 4 Dec 2001 10:49:32 +0900')
    assert_equal 'Tue, 4 Dec 2001 10:49:32 +0900', h.to_s
    assert_equal h.to_s, h.decoded
    ok = h.to_s

    h = TMail::HeaderField.new('Date', 'Tue, 4 Dec 2001 01:49:32 +0000')
    assert_equal ok, h.to_s

    h = TMail::HeaderField.new('Date', 'Tue, 4 Dec 2001 01:49:32 GMT')
    assert_equal ok, h.to_s
  end

end


class AddressHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( To Cc Bcc From Reply-To
        Resent-To Resent-Cc Resent-Bcc
        Resent-From Resent-Reply-To ).each do |name|
      h = TMail::HeaderField.new(name, 'aamine@loveruby.net')
      assert_instance_of TMail::AddressHeader, h, name
    end
  end

  def validate_case( str, isempty, to_s, comments, succ )
    h = TMail::HeaderField.new('To', str)
    assert_equal isempty, h.empty?,            str.inspect + " (empty?)\n"
    assert_instance_of Array, h.addrs,         str.inspect + " (is a)\n"
    assert_equal succ.size, h.addrs.size,      str.inspect + " (size)\n"
    h.addrs.each do |a|
      ok = succ.shift
      assert_equal ok[:phrase],   a.phrase,    str.inspect + " (phrase)\n"
      assert_equal ok[:routes],   a.routes,    str.inspect + " (routes)\n"
      assert_equal ok[:spec],     a.spec,      str.inspect + " (spec)\n"
    end
    assert_equal comments, h.comments,         str.inspect + " (comments)\n"
    assert_equal to_s, h.to_s,                 str.inspect + " (to_s)\n" if to_s
    assert_equal to_s, h.decoded,              str.inspect + " (decoded)\n" if to_s
  end
    
  def test_ATTRS
    validate_case 'aamine@loveruby.net',
        false,
        'aamine@loveruby.net',
        [],
    [{  :phrase   => nil,
        :routes   => [],
        :spec     => 'aamine@loveruby.net' }]

    validate_case 'Minero Aoki <aamine@loveruby.net> (comment)',
        false,
        'Minero Aoki <aamine@loveruby.net> (comment)',
        ['comment'],
    [{  :phrase   => 'Minero Aoki',
        :routes   => [],
        :spec     => 'aamine@loveruby.net' }]

    validate_case 'aamine@loveruby.net, , taro@softica.org',
        false,
        'aamine@loveruby.net, taro@softica.org',
        [],
    [{  :phrase   => nil,
        :routes   => [],
        :spec     => 'aamine@loveruby.net' },
     {  :phrase   => nil,
        :routes   => [],
        :spec     => 'taro@softica.org' }]

    validate_case '',
        true,
        nil,
        [],
    []

    validate_case '(comment only)',
        true,
        nil,
        ['comment only'],
    []
  end

end


class SingleAddressHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    h = TMail::HeaderField.new('Sender', 'aamine@loveruby.net')
    assert_instance_of TMail::SingleAddressHeader, h
  end

  def test_addr
    h = TMail::HeaderField.new('Sender', 'aamine@loveruby.net')
    assert_not_nil h.addr
    assert_instance_of TMail::Address, h.addr
    assert_equal 'aamine@loveruby.net', h.addr.spec
    assert_equal nil, h.addr.phrase
    assert_equal [], h.addr.routes
  end

  def test_to_s
    str = 'Minero Aoki <aamine@loveruby.net>, "AOKI, Minero" <aamine@softica.org>'
    h = TMail::HeaderField.new('Sender', str)
    assert_equal 'Minero Aoki <aamine@loveruby.net>', h.to_s
  end

end


class ReturnPathHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Return-Path ).each do |name|
      h = TMail::HeaderField.new(name, '<aamine@loveruby.net>')
      assert_instance_of TMail::ReturnPathHeader, h, name
      assert_equal false, h.empty?
      assert_equal false, h.illegal?
    end
  end

  def test_ATTRS
    h = TMail::HeaderField.new('Return-Path', '<@a,@b,@c:aamine@loveruby.net>')
    assert_not_nil h.addr
    assert_instance_of TMail::Address, h.addr
    assert_equal 'aamine@loveruby.net', h.addr.spec
    assert_equal nil, h.addr.phrase
    assert_equal ['a', 'b', 'c'], h.addr.routes

    assert_not_nil h.routes
    assert_instance_of Array, h.routes
    assert_equal ['a', 'b', 'c'], h.routes
    assert_equal h.addr.routes, h.routes

    assert_not_nil h.spec
    assert_instance_of String, h.spec
    assert_equal 'aamine@loveruby.net', h.spec

    # missing '<' '>'
    h = TMail::HeaderField.new('Return-Path', 'xxxx@yyyy')
    assert_equal 'xxxx@yyyy', h.spec
  end

  def test_to_s
    body = 'Minero Aoki <@a,@b,@c:aamine@loveruby.net>'
    h = TMail::HeaderField.new('Return-Path', body)
    assert_equal '<@a,@b,@c:aamine@loveruby.net>', h.to_s
    assert_equal h.to_s, h.decoded

    body = 'aamine@loveruby.net'
    h = TMail::HeaderField.new('Return-Path', body)
    assert_equal '<aamine@loveruby.net>', h.to_s
    assert_equal h.to_s, h.decoded
  end

end


class MessageIdHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Message-Id MESSAGE-ID Message-ID
        Resent-Message-Id Content-Id ).each do |name|
      h = TMail::HeaderField.new(name, '<20020103xg88.k0@mail.loveruby.net>')
      assert_instance_of TMail::MessageIdHeader, h
    end
  end

  def test_id
    str = '<20020103xg88.k0@mail.loveruby.net>'
    h = TMail::HeaderField.new('Message-Id', str)
    assert_not_nil h.id
    assert_equal str, h.id

    id = '<20020103xg88.k0@mail.loveruby.net>'
    str = id + ' (comm(ent))'
    h = TMail::HeaderField.new('Message-Id', str)
    assert_not_nil h.id
    assert_equal id, h.id
  end

  def test_id=
    h = TMail::HeaderField.new('Message-Id', '')
    h.id = str = '<20020103xg88.k0@mail.loveruby.net>'
    assert_not_nil h.id
    assert_equal str, h.id
  end

end


class ReferencesHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    str = '<20020103xg88.k0@mail.loveruby.net>'
    %w( References REFERENCES ReFeReNcEs
        In-Reply-To ).each do |name|
      h = TMail::HeaderField.new(name, str)
      assert_instance_of TMail::ReferencesHeader, h, name
    end
  end

  def test_ATTRS
    id1 = '<20020103xg88.k0@mail.loveruby.net>'
    id2 = '<20011204103415.64DB.GGB03124@nifty.ne.jp>'
    phr = 'message of "Wed, 17 Mar 1999 18:42:07 +0900"'
    str = id1 + ' ' + phr + ' ' + id2

    h = TMail::HeaderField.new('References', str)

    ok = [id1, id2]
    h.each_id do |i|
      assert_equal ok.shift, i
    end
    ok = [id1, id2]
    assert_equal ok, h.ids
    h.each_id do |i|
      assert_equal ok.shift, i
    end

    ok = [phr]
    assert_equal ok, h.phrases
    h.each_phrase do |i|
      assert_equal ok.shift, i
    end
    ok = [phr]
    h.each_phrase do |i|
      assert_equal ok.shift, i
    end


    # test 2
    # 'In-Reply-To'
    # 'aamine@dp.u-netsurf.ne.jp's message of "Fri, 8 Jan 1999 03:49:37 +0900"'
  end

  def test_to_s
    id1 = '<20020103xg88.k0@mail.loveruby.net>'
    id2 = '<20011204103415.64DB.GGB03124@nifty.ne.jp>'
    phr = 'message of "Wed, 17 Mar 1999 18:42:07 +0900"'
    str = id1 + ' ' + phr + ' ' + id2

    h = TMail::HeaderField.new('References', str)
    assert_equal id1 + ' ' + id2, h.to_s
  end

end


class ReceivedHeaderTester < RUNIT::TestCase

  testme!

  HEADER1 = <<EOS
from helium.ruby-lang.org (helium.ruby-lang.org [210.251.121.214])
	by doraemon.edit.ne.jp (8.12.1/8.12.0) via TCP with ESMTP
        id fB41nwEj007438 for <aamine@mx.edit.ne.jp>;
        Tue, 4 Dec 2001 10:49:58 +0900 (JST)
EOS
  HEADER2 = <<EOS
from helium.ruby-lang.org (localhost [127.0.0.1])
	by helium.ruby-lang.org (Postfix) with ESMTP
	id 8F8951AF3F; Tue,  4 Dec 2001 10:49:32 +0900 (JST)
EOS
  HEADER3 = <<EOS
from smtp1.dti.ne.jp (smtp1.dti.ne.jp [202.216.228.36])
	by helium.ruby-lang.org (Postfix) with ESMTP id CE3A1C3
	for <ruby-list@ruby-lang.org>; Tue,  4 Dec 2001 10:49:31 +0900 (JST)
EOS

=begin  dangerous headers
# 2-word WITH (this header is also wrong in semantic)
# I cannot support this.
Received: by mebius with Microsoft Mail
	id <01BE2B9D.9051EAA0@mebius>; Sat, 19 Dec 1998 22:18:54 -0800
=end

  def test_s_new
    %w( Received ).each do |name|
      h = TMail::HeaderField.new(name, HEADER1)
      assert_instance_of TMail::ReceivedHeader, h, name
    end
  end

  def test_ATTRS
    h = TMail::HeaderField.new('Received', HEADER1)
    assert_instance_of String, h.from
    assert_equal 'helium.ruby-lang.org', h.from

    assert_instance_of String, h.by
    assert_equal 'doraemon.edit.ne.jp', h.by

    assert_instance_of String, h.via
    assert_equal 'TCP', h.via

    assert_instance_of Array, h.with
    assert_equal %w(ESMTP), h.with

    assert_instance_of String, h.id
    assert_equal 'fB41nwEj007438', h.id

    assert_instance_of String, h._for
    assert_equal 'aamine@mx.edit.ne.jp', h._for   # must be <a> ?

    assert_instance_of Time, h.date
    assert_equal Time.local(2001,12,4, 10,49,58), h.date

    h = TMail::HeaderField.new('Received', '; Tue, 4 Dec 2001 10:49:58 +0900')
    assert_nil h.from
    assert_nil h.by
    assert_nil h.via
    assert_equal [], h.with
    assert_nil h.id
    assert_nil h._for
    assert_equal Time.local(2001,12,4, 10,49,58), h.date

    # without date
    h = TMail::HeaderField.new('Received', 'by NeXT.Mailer (1.144.2)')
    assert_nil h.from
    assert_equal 'NeXT.Mailer', h.by
    assert_nil h.via
    assert_equal [], h.with
    assert_nil h.id
    assert_nil h._for
    assert_nil h.date

    # FROM is not a domain
    h = TMail::HeaderField.new('Received',
        'from someuser@example.com; Tue, 24 Nov 1998 07:59:39 -0500')
    assert_equal 'example.com', h.from
    assert_nil h.by
    assert_nil h.via
    assert_equal [], h.with
    assert_nil h.id
    assert_nil h._for
    assert_equal Time.local(1998,11,24, 7+5+9,59,39), h.date

=begin
    # FOR is not route-addr.
    # item order is wrong.
    h = TMail::HeaderField.new('Received',
        'from aamine by mail.softica.org with local for list@softica.org id 12Vm3N-00044L-01; Fri, 17 Mar 2000 10:59:53 +0900')
    assert_equal 'aamine', h.from
    assert_equal 'mail.softica.org', h.by
    assert_nil h.via
    assert_equal ['local'], h.with
    assert_equal '12Vm3N-00044L-01', h.id
    assert_equal 'list@softica.org', h._for
    assert_equal Time.local(2000,4,17, 10,59,53), h.date
=end

    # word + domain-literal in FROM
    h = TMail::HeaderField.new('Received',
        'from localhost [192.168.1.1]; Sat, 19 Dec 1998 22:19:50 PST')
    assert_equal 'localhost', h.from
    assert_nil h.by
    assert_nil h.via
    assert_equal [], h.with
    assert_nil h.id
    assert_nil h._for
    assert_equal Time.local(1998,12,19+1, 22+8+9-24,19,50), h.date

    # addr-spec in BY (must be a domain)
    h = TMail::HeaderField.new('Received',
        'by aamine@loveruby.net; Wed, 24 Feb 1999 14:34:20 +0900')
    assert_equal 'loveruby.net', h.by
  end

  def test_to_s
    h = TMail::HeaderField.new('Received', HEADER1)
    assert_equal 'from helium.ruby-lang.org by doraemon.edit.ne.jp via TCP with ESMTP id fB41nwEj007438 for <aamine@mx.edit.ne.jp>; Tue, 4 Dec 2001 10:49:58 +0900', h.to_s

    [
      'from harmony.loveruby.net',
      'by mail.loveruby.net',
      'via TCP',
      'with ESMTP',
      'id LKJHSDFG',
      'for <aamine@loveruby.net>',
      '; Tue, 4 Dec 2001 10:49:58 +0900'
    ]\
    .each do |str|
      h = TMail::HeaderField.new('Received', str)
      assert_equal str, h.to_s, 'ReceivedHeader#to_s: data=' + str.dump
    end
  end

end


class KeywordsHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Keywords KEYWORDS KeYwOrDs ).each do |name|
      h = TMail::HeaderField.new(name, 'key, word, is, keyword')
      assert_instance_of TMail::KeywordsHeader, h
    end
  end

  def test_keys
    h = TMail::HeaderField.new('Keywords', 'key, word, is, keyword')
    assert_instance_of Array, h.keys
    assert_equal %w(key word is keyword), h.keys
  end

end


class EncryptedHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Encrypted ).each do |name|
      h = TMail::HeaderField.new(name, 'lot17 solt')
      assert_instance_of TMail::EncryptedHeader, h
    end
  end

  def test_encrypter
    h = TMail::HeaderField.new('Encrypted', 'lot17 solt')
    assert_equal 'lot17', h.encrypter
  end

  def test_encrypter=
    h = TMail::HeaderField.new('Encrypted', 'lot17 solt')
    h.encrypter = 'newscheme'
    assert_equal 'newscheme', h.encrypter
  end

  def test_keyword
    h = TMail::HeaderField.new('Encrypted', 'lot17 solt')
    assert_equal 'solt', h.keyword
    h = TMail::HeaderField.new('Encrypted', 'lot17')
    assert_equal nil, h.keyword
  end

  def test_keyword=
    h = TMail::HeaderField.new('Encrypted', 'lot17 solt')
    h.keyword = 'newscheme'
    assert_equal 'newscheme', h.keyword
  end

end


class MimeVersionHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Mime-Version MIME-VERSION MiMe-VeRsIoN ).each do |name|
      h = TMail::HeaderField.new(name, '1.0')
      assert_instance_of TMail::MimeVersionHeader, h
    end
  end

  def test_ATTRS
    h = TMail::HeaderField.new('Mime-Version', '1.0')
    assert_equal 1, h.major
    assert_equal 0, h.minor
    assert_equal '1.0', h.version

    h = TMail::HeaderField.new('Mime-Version', '99.77 (is ok)')
    assert_equal 99, h.major
    assert_equal 77, h.minor
    assert_equal '99.77', h.version
  end

  def test_major=
    h = TMail::HeaderField.new('Mime-Version', '1.1')
    h.major = 2
    assert_equal 2, h.major
    assert_equal 1, h.minor
    assert_equal 2, h.major
    h.major = 3
    assert_equal 3, h.major
  end

  def test_minor=
    h = TMail::HeaderField.new('Mime-Version', '2.3')
    assert_equal 3, h.minor
    h.minor = 5
    assert_equal 5, h.minor
    assert_equal 2, h.major
  end

  def test_to_s
    h = TMail::HeaderField.new('Mime-Version', '1.0 (first version)')
    assert_equal '1.0', h.to_s
  end

  def test_empty?
    h = TMail::HeaderField.new('Mime-Version', '')
    assert_equal true, h.empty?
  end

end


class ContentTypeHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Content-Type CONTENT-TYPE CoNtEnT-TyPe ).each do |name|
      h = TMail::HeaderField.new(name, 'text/plain; charset=iso-2022-jp')
      assert_instance_of TMail::ContentTypeHeader, h, name
    end
  end

  def test_ATTRS
    h = TMail::HeaderField.new('Content-Type', 'text/plain; charset=iso-2022-jp')
    assert_equal 'text', h.main_type
    assert_equal 'plain', h.sub_type
    assert_equal 1, h.params.size
    assert_equal 'iso-2022-jp', h.params['charset']

    h = TMail::HeaderField.new('Content-Type', 'Text/Plain; Charset=shift_jis')
    assert_equal 'text', h.main_type
    assert_equal 'plain', h.sub_type
    assert_equal 1, h.params.size
    assert_equal 'shift_jis', h.params['charset']

    h = TMail::HeaderField.new('Content-Type', 'multipart/mixed; boundary=dDRMvlgZJXvWKvBx')
    assert_equal 'multipart', h.main_type
    assert_equal 'mixed', h.sub_type
    assert_equal 1, h.params.size
    assert_equal 'dDRMvlgZJXvWKvBx', h.params['boundary']
  end

  def test_main_type=
    h = TMail::HeaderField.new('Content-Type', 'text/plain; charset=iso-2022-jp')
    assert_equal 'text', h.main_type
    h.main_type = 'multipart'
    assert_equal 'multipart', h.main_type
    assert_equal 'multipart', h.main_type

    h.main_type = 'TEXT'
    assert_equal 'text', h.main_type
  end

  def test_sub_type=
    h = TMail::HeaderField.new('Content-Type', 'text/plain; charset=iso-2022-jp')
    assert_equal 'plain', h.sub_type
    h.sub_type = 'html'
    assert_equal 'html', h.sub_type
    h.sub_type = 'PLAIN'
    assert_equal 'plain', h.sub_type
  end

end


class ContentEncodingHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Content-Transfer-Encoding CONTENT-TRANSFER-ENCODING
        COnteNT-TraNSFer-ENCodiNG ).each do |name|
      h = TMail::HeaderField.new(name, 'Base64')
      assert_instance_of TMail::ContentTransferEncodingHeader, h
    end
  end

  def test_encoding
    h = TMail::HeaderField.new('Content-Transfer-Encoding', 'Base64')
    assert_equal 'base64', h.encoding
    
    h = TMail::HeaderField.new('Content-Transfer-Encoding', '7bit')
    assert_equal '7bit', h.encoding
  end

  def test_encoding=
    h = TMail::HeaderField.new('Content-Transfer-Encoding', 'Base64')
    assert_equal 'base64', h.encoding
    h.encoding = '7bit'
    assert_equal '7bit', h.encoding
  end

  def test_to_s
    h = TMail::HeaderField.new('Content-Transfer-Encoding', 'Base64')
    assert_equal 'Base64', h.to_s
    assert_equal h.to_s, h.decoded
    assert_equal h.to_s, h.encoded
  end

end


class ContentDispositionHeaderTester < RUNIT::TestCase

  testme!

  def test_s_new
    %w( Content-Disposition CONTENT-DISPOSITION
        ConTENt-DIsPOsition ).each do |name|
      h = TMail::HeaderField.new(name, 'attached; filename="README.txt.pif"')
      assert_instance_of TMail::ContentDispositionHeader, h
    end
  end

  def test_ATTRS
    begin
      $KCODE = 'NONE'
      _test_ATTRS
    ensure
      $KCODE = 'NONE'
    end
  end

  def _test_ATTRS
    h = TMail::HeaderField.new('Content-Disposition',
                               'attached; filename="README.txt.pif"')
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal 'README.txt.pif', h.params['filename']

    h = TMail::HeaderField.new('Content-Disposition',
                               'Attached; Filename="README.txt.pif"')
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal 'README.txt.pif', h.params['filename']

    h = TMail::HeaderField.new('Content-Disposition',
                               'attached; filename=')
    assert_equal true, h.empty?
    assert_nil h.params
    assert_nil h['filename']


    $KCODE = 'EUC'
    # raw iso2022jp string in value (token)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename=\e$BF|K\\8l\e(B.doc>)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    # assert_equal "\e$BF|K\\8l\e(B.doc", h.params['filename']
    assert_equal "\306\374\313\334\270\354.doc", h.params['filename']

    # raw iso2022jp string in value (quoted string)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename="\e$BF|K\\8l\e(B.doc">)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    # assert_equal "\e$BF|K\\8l\e(B.doc", h.params['filename']
    assert_equal "\306\374\313\334\270\354.doc", h.params['filename']


    $KCODE = 'EUC'
    # RFC2231 decode
    h = TMail::HeaderField.new('Content-Disposition',
            "attached; filename*=iso-2022-jp'ja'%1b$B$Q$i$`%1b%28B")
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\244\321\244\351\244\340", h.params['filename']

    # raw EUC-JP string in value (token)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename=\306\374\313\334\270\354.doc>)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\306\374\313\334\270\354.doc", h.params['filename']

    # raw EUC-JP string in value (quoted-string)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename="\306\374\313\334\270\354.doc">)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\306\374\313\334\270\354.doc", h.params['filename']


    $KCODE = 'SJIS'
    # raw SJIS string in value (token)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename=\223\372\226{\214\352.doc>)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\223\372\226{\214\352.doc", h.params['filename']

    # raw SJIS string in value (quoted-string)
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename="\223\372\226{\214\352.doc">)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\223\372\226{\214\352.doc", h.params['filename']


    # JIS -> $KCODE auto conversion
    $KCODE = 'EUC'
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename=\e$BF|K\\8l\e(B.doc>)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\306\374\313\334\270\354.doc", h.params['filename']

    $KCODE = 'SJIS'
    h = TMail::HeaderField.new('Content-Disposition',
            %Q<attached; filename=\e$BF|K\\8l\e(B.doc>)
    assert_equal 'attached', h.disposition
    assert_equal 1, h.params.size
    assert_equal "\223\372\226{\214\352.doc", h.params['filename']
  end

  def test_disposition=
    h = TMail::HeaderField.new('Content-Disposition',
                               'attached; filename="README.txt.pif"')
    assert_equal 'attached', h.disposition
    h.disposition = 'virus'
    assert_equal 'virus', h.disposition
    h.disposition = 'AtTacHed'
    assert_equal 'attached', h.disposition
  end

end
