
require 'fileutils'
require 'assert'


class FileUtilsTester < RUNIT::TestCase

  testme!

  include FileUtils


  def setup
    system 'rm -rf data; mkdir data'
    system 'rm -rf tmp;  mkdir tmp'
    prepare_data_file
    prepare_time_data
  end

  def teardown
    system 'rm -rf data'
    system 'rm -rf tmp'
  end


  DATA = %w( data/1 data/2 data/3 )

  ALLCHAR = s = ''
  (0..255).each {|i| s.concat i }

  UNIQLINE = s = ''
  srand
  50.times { s << rand(255) }

  def prepare_data_file
    File.open( 'data/1', 'w' ) {|f|
      32.times do
        f.puts 'a' * 50
      end
    }
    File.open( 'data/2', 'w' ) {|f|
      32.times do
        f.puts ALLCHAR
      end
    }
    File.open( 'data/3', 'w' ) {|f|
      32.times do
        f.puts UNIQLINE
      end
    }
  end

  BIGFILE = 'data/big'

  def prepare_big_file
    File.open( 'data/big', 'w' ) {|f|
      s = ''
      (0..255).each {|i| s.concat i }
      (4 * 1024 * 1024 / 256).times do   # 4MB
        f.puts 'aaaa aaaa aaaa aaaa aaaa aaaa aaaa aaaa aaaa aaaa'
      end
    }
  end

  def prepare_time_data
    File.open( 'data/old',    'w' ) {|f| f.puts 'dummy' }
    File.open( 'data/newer',  'w' ) {|f| f.puts 'dummy' }
    File.open( 'data/newest', 'w' ) {|f| f.puts 'dummy' }
    t = Time.now
    File.utime t-8, t-8, 'data/old'
    File.utime t-4, t-4, 'data/newer'
  end


  def test_cmp
    DATA.each do |fname|
      assert cmp(fname, fname), 'not same?'
    end
  end

  def test_cp
    DATA.each do |fname|
      cp fname, 'tmp/cp'
      assert_same_file fname, 'tmp/cp'

      cp fname, 'tmp'
      assert_same_file fname, 'tmp/' + File.basename(fname)

      cp fname, 'tmp/preserve', :preserve
      assert_same_file fname, 'tmp/preserve'
      a = File.stat(fname)
      b = File.stat('tmp/preserve')
      assert_equal a.mode, b.mode
      assert_equal a.mtime, b.mtime
      assert_equal a.uid, b.uid
      assert_equal a.gid, b.gid
    end
  end

  def test_mv
    DATA.each do |fname|
      cp fname, 'tmp/mvsrc'
      mv 'tmp/mvsrc', 'tmp/mvdest'
      assert_same_file fname, 'tmp/mvdest'
    end
  end

  def test_rm
    DATA.each do |fname|
      cp fname, 'tmp/rmsrc'
      rm 'tmp/rmsrc'
      assert_file_not_exist 'tmp/rmsrc'
    end
  end

  def test_rm_f
    DATA.each do |fname|
      cp fname, 'tmp/rmsrc'
      rm_f 'tmp/rmsrc'
      assert_file_not_exist 'tmp/rmsrc'
    end

    File.open( 'tmp/lnf_symlink_src', 'w' ) {|f| f.puts 'dummy' }
    File.symlink 'tmp/lnf_symlink_src', 'tmp/lnf_symlink_dest'
    rm_f 'tmp/lnf_symlink_dest'
    assert_file_not_exist 'tmp/lnf_symlink_dest'
    assert_file_exist     'tmp/lnf_symlink_src'

    rm_f 'notexistdatafile'
    rm_f 'tmp/notexistdatafile'
    system 'rm -rf tmpdatadir'
    Dir.mkdir 'tmpdatadir'
    rm_f 'tmpdatadir'
    Dir.rmdir 'tmpdatadir'
  end

  def test_rm_r
    system 'rm -rf tmpdatadir'

    Dir.mkdir 'tmpdatadir'
    rm_r 'tmpdatadir'
    assert_file_not_exist 'tmpdatadir'

    Dir.mkdir 'tmpdatadir'
    rm_r 'tmpdatadir/'
    assert_file_not_exist 'tmpdatadir'

    Dir.mkdir 'tmp/tmpdir'
    rm_r 'tmp/tmpdir/'
    assert_file_not_exist 'tmp/tmpdir'
    assert_file_exist     'tmp'

    Dir.mkdir 'tmp/tmpdir'
    rm_r 'tmp/tmpdir'
    assert_file_not_exist 'tmp/tmpdir'
    assert_file_exist     'tmp'

    Dir.mkdir 'tmp/tmpdir'
    File.open( 'tmp/tmpdir/a', 'w' ) {|f| f.puts 'dummy' }
    File.open( 'tmp/tmpdir/b', 'w' ) {|f| f.puts 'dummy' }
    File.open( 'tmp/tmpdir/c', 'w' ) {|f| f.puts 'dummy' }
    rm_r 'tmp/tmpdir'
    assert_file_not_exist 'tmp/tmpdir'
    assert_file_exist     'tmp'
  end

  def test_with_big_file
    prepare_big_file

    cp BIGFILE, 'tmp/cpdest'
    assert_same_file BIGFILE, 'tmp/cpdest'
    assert cmp(BIGFILE, 'tmp/cpdest'), 'orig != copied'

    mv 'tmp/cpdest', 'tmp/mvdest'
    assert_same_file BIGFILE, 'tmp/mvdest'
    assert_file_not_exist 'tmp/cpdest'

    rm 'tmp/mvdest'
    assert_file_not_exist 'tmp/mvdest'
  end

  def test_ln
    DATA.each do |fname|
      ln fname, 'tmp/lndest'
      assert_same_file fname, 'tmp/lndest'
      File.unlink 'tmp/lndest'
    end

    ln DATA, 'tmp'
    DATA.each do |fname|
      assert_same_file fname, 'tmp/' + File.basename(fname)
    end
    DATA.each do |fname|
      File.unlink 'tmp/' + File.basename(fname)
    end
  end

  def test_ln_s
    DATA.each do |fname|
      ln_s fname, 'tmp/lnsdest'
      assert FileTest.symlink?('tmp/lnsdest'), 'not symlink'
      assert_equal fname, File.readlink('tmp/lnsdest')
      rm_f 'tmp/lnsdest'
    end
  end

  def test_ln_sf
    DATA.each do |fname|
      ln_sf fname, 'tmp/lnsdest'
      assert FileTest.symlink?('tmp/lnsdest'), 'not symlink'
      assert_equal fname, File.readlink('tmp/lnsdest')
      ln_sf fname, 'tmp/lnsdest'
      ln_sf fname, 'tmp/lnsdest'
    end
  end

  def test_mkdir
    system 'rm -rf tmpdatadir'
    mkdir 'tmpdatadir'
    assert_is_directory 'tmpdatadir'
    Dir.rmdir 'tmpdatadir'

    mkdir 'tmp/mkdirdest'
    assert_is_directory 'tmp/mkdirdest'
    Dir.rmdir 'tmp/mkdirdest'
  end

  def test_mkdir_p
    dirs = %w(
      tmpdir/dir/
      tmpdir/dir/./
      tmpdir/dir/./.././dir/
      tmpdir/a/b/c/
      tmpdir/a/b/c
      tmpdir/a/b
      tmpdir/a/b/
      tmpdir/a
      tmpdir/a/
    )
    rm_rf 'tmpdir'
    try_mkdirp dirs, true
    try_mkdirp dirs, false
    rm_rf 'tmpdir'
  end

  def try_mkdirp( dirs, del )
    dirs.each do |d|
      mkdir_p d
      assert_is_directory d
      rm_rf 'tmpdir' if del
    end
  end

  def test_uptodate?
    Dir.chdir('data') {
      assert(   uptodate?('newest', 'old', 'newer', 'notexist') )
      assert( ! uptodate?('newer', 'old', 'newest', 'notexist') )
      assert( ! uptodate?('notexist', 'old', 'newest', 'newer') )
    }
  end

end
