#
# mail.rb
#
#   Copyright (c) 1998-2001 Minero Aoki <aamine@dp.u-netsurf.ne.jp>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU Lesser General Public License version 2 or later.
#

require 'nkf'
require 'socket'

require 'amstd/to_s'
require 'amstd/bug'

require 'tmail/field'
require 'tmail/port'
require 'tmail/encode'


module TMail

  class MailError < StandardError; end
  class MailSyntaxError < MailError; end


  class << self

    def new_boundary
      'mimepart_' + random_tag
    end

    alias boundary new_boundary

    def new_msgid( fqdn = nil )
      domain = (fqdn && "#{fqdn}.tmail") || ::Socket.gethostname
      "<#{random_tag}@#{domain}>"
    end

    alias msgid new_msgid

    private

    def random_tag
      @uniq += 1
      t = Time.now
      sprintf( '%x%x_%x%x%d%x',
               t.to_i,
               t.tv_usec,
               $$,
               Thread.current.id,
               @uniq,
               rand(255) )
    end

  end

  @uniq = 0


  class Mail

    class << self

      def load_from( fname )
        new FilePort.new( fname )
      end

      alias loadfrom load_from

      def parse( str )
        new StringPort.new( str )
      end

      # for compatibility

      def boundary
        TMail.new_boundary
      end

      def msgid
        TMail.new_msgid
      end

    end


    def initialize( port = nil, strict = false )
      @port = port || StringPort.new('')
      @strict = strict

      @header    = {}
      @body_port = nil
      @epilogue  = ''
      @parts     = []

      parse_header
    end

    attr :port

    def inspect
      "\#<#{type} port=#{@port.inspect} bodyport=#{@body_port.inspect}>"
    end


    def encoded( eol = "\r\n", charset = 'j', ret = '', sep = '' )
      en = HFencoder.new( ret, eol, charset )
      accept en, eol, ret, sep
      ret
    end

    def decoded( eol = "\n", charset = 'e', ret = '', sep = '' )
      stra = HFdecoder.new( ret, charset )
      accept stra, eol, ret, sep
      ret
    end

    alias to_s decoded

    def accept( stra, eol = "\n", ret = '', sep = '' )
      multipart_p = ! parts.empty?    # DO NOT USE @parts

      if multipart_p then
        bound = ::TMail.new_boundary
        h = self['content-type']
        if h then
          h.params['boundary'] = bound
        else
          store 'Content-Type', %<multipart/mixed; boundary="#{bound}">
        end
      end

      header_accept stra, ret, eol, sep

      body_port.ropen do |is|
        is.each do |line|
          line.sub!( /[\r\n]+\z/, '' )
          ret << line << eol
        end
      end

      if multipart_p then
        parts.each do |tm|
          ret << "#{eol}--#{bound}#{eol}"
          tm.accept( stra, eol, ret, sep )
        end
        ret << "#{eol}--#{bound}--#{eol}"
        s = epilogue
        ret << s
        case s[-1] when ?\n then; when ?\r then; else
          ret << eol
        end
      end
    end


    def write_back( eol = "\n", charset = 'e', sep = '' )
      @port.wopen {|os| encoded eol, charset, os, sep }
    end



    ###
    ### header
    ###


    USE_ARRAY = %w(
      received
    )

    def header
      @header.dup
    end


    def fetch( key, initbody = nil, &block )
      dkey = key.downcase
      ret = @header[ dkey ]

      unless ret then
        if iterator? then
          initbody = yield
        end
        if initbody then
          ret = mkhf( *ret.split(':', 2) )
          do_store dkey, ret
        end
      end

      ret
    end

    def store( key, val )
      dkey = key.downcase

      if val.nil? then
        @header.delete dkey
        return
      end

      case val
      when String
        val = mkhf( key, val )
      when HeaderField
        ;
      when Array
        unless USE_ARRAY.include? dkey then
          raise ArgumentError, "#{key}: Header must not be multiple"
        end
        @header[dkey] = val
        return
      else
        val = mkhf( key, val.to_s )
      end

      if USE_ARRAY.include? dkey then
        if arr = @header[dkey] then
          arr.push val
        else
          @header[dkey] = [val]
        end
      else
        @header[dkey] = val
      end
    end

    def []( key )
      @header[ key.downcase ]
    end

    alias []= store


    def each_header( &block )
      @header.each {|k,v| yield_h k, v, &block }
    end

    alias each_pair each_header

    def each_header_name( &block )
      @header.each_key( &block )
    end

    alias each_key each_header_name

    def each_field( &block )
      @header.each_key do |key|
        v = fetch( key )
        if Array === v then
          v.each( &block )
        else
          yield v
        end
      end
    end

    alias each_value each_field

    FIELD_ORDER = %w(
      received date from sender reply-to in-reply-to subject
      to cc bcc message-id references
      mime-version content-type content-transfer-encoding
      content-disposition content-description
    )

    def ordered_each( &block )
      tmp = @header.dup
      FIELD_ORDER.each do |name|
        if f = tmp.delete(name) then
          yield_h name, f, &block
        end
      end
      tmp.each {|name, f| yield_h name, f, &block }
    end


    def clear
      @header.clear
    end

    def delete( key )
      @header.delete key.downcase
    end

    def delete_if
      @header.delete_if do |key,v|
        v = fetch( key )
        if Array === v then
          v.delete_if{|f| yield key, f }
          v.empty?
        else
          yield key, v
        end
      end
    end


    def keys
      @header.keys
    end

    def has_key?( key )
      @header.has_key? key.downcase
    end

    alias include? has_key?
    alias key?     has_key?


    def values
      ret = []
      each_field {|v| ret.push v }
      ret
    end

    def has_value?( val )
      return false unless HeaderField === val

      v = fetch( val.name )
      if Array === v then v.include? val
      else                v ? (val == v) : false
      end
    end

    alias value? has_value?


    def indexes( *args )
      args.collect {|k| fetch k }.flatten
    end

    alias indices indexes


    private


    def yield_h( name, field )
      if Array === field then
        field.each do |f|
          yield name, f
        end
      else
        yield name, field
      end
    end


    def header_accept( strategy, ret, eol, sep )
      ordered_each do |name, header|
        header.accept strategy
        strategy.write_in
        ret << eol
      end
      ret << sep << eol
    end


    def mkhf( fname, fbody )
      fname.strip!
      HeaderField.new( fname, fbody, @strict )
    end

    def parse_header
      fname = fbody = nil
      unixfrom = nil
      errlog = []

      begin
        @stream = src = @port.ropen

        while line = src.gets do     # no each !
          case line
          when /\A[ \t]/             # continue from prev line
            unless fbody then
              errlog.push 'mail is began by space or tab'
              next
            end
            line.strip!
            fbody << ' ' << line
            # fbody << line

          when /\A([^\: \t]+):\s*/   # new header line
            add_hf fname, fbody if fbody
            fname = $1
            fbody = $'
            # fbody.strip!

          when /\A\-*\s*\z/          # end of header
            add_hf fname, fbody if fbody
            fname = fbody = nil
            break

          when /\AFrom (\S+)/
            unixfrom = $1

          else
            errlog.push "wrong mail header: '#{line.inspect}'"
          end
        end
        add_hf fname, fbody if fname
      ensure
        src.stop if src
      end
      errlog.empty? or raise MailSyntaxError, "\n" + errlog.join("\n")

      if unixfrom then
        add_hf 'Return-Path', "<#{unixfrom}>" unless @header['return-path']
      end
    end

    def add_hf( fname, fbody )
      key = fname.downcase
      hf = mkhf( fname, fbody )

      if USE_ARRAY.include? key then
        if tmp = @header[key] then
          tmp.push hf
        else
          @header[key] = [hf]
        end
      else
        @header[key] = hf
      end
    end



    ###
    ### body
    ###

    public


    def body_port
      parse_body
      @body_port
    end

    def each
      body_port.ropen do |f|
        f.each {|line| yield line }
      end
    end

    def body
      parse_body
      ret = nil
      @body_port.ropen {|is| ret = is.read_all }
      ret
    end

    def body=( str )
      parse_body
      @body_port.wopen {|os| os.write str }
      true
    end

    alias preamble  body
    alias preamble= body=

    def epilogue
      parse_body
      @epilogue.dup
    end

    def epilogue=( str )
      parse_body
      @epilogue = str
      str
    end

    def parts
      parse_body
      @parts
    end


    private


    def parse_body
      if @stream then
        parse_body_in @stream
        @stream = nil
      end
    end
    
    def parse_body_in( stream )
      begin
        stream.restart
        if multipart? then
          read_multipart stream
        else
          @body_port = tmp_port
          @body_port.wopen do |f|
            stream.copy_to f
          end
        end
      ensure
        stream.close unless stream.closed?
      end
    end

    def read_multipart( src )
      bound = self['content-type'].params['boundary']
      is_sep = /\A--#{Regexp.quote bound}(?:--)?[ \t]*(?:\n|\r\n|\r)/
      lastbound = "--#{bound}--"

      f = nil
      n = 1
      begin
        ports = []
        ports.push tmp_port
        f = ports[-1].wopen

        while line = src.gets do    # no each !
          if is_sep === line then
            f.close
            line.strip!
            if line == lastbound then
              break
            else
              ports.push tmp_port n
              n += 1
              f = ports[-1].wopen
            end
          else
            f << line
          end
        end
        @epilogue = src.read_all
      ensure
        f.close if f and not f.closed?
      end

      @body_port = ports.shift
      @parts = ports.collect {|p| type.new p }
    end

    def tmp_port( n = 0 )
      StringPort.new ''
    end



    ###
    ### indirect access methods
    ###

    public


    class << self
    
      def tmail_attr( mname, hname, part )
        module_eval %-
          def #{mname.id2name}( default = nil )
            if header = @header['#{hname}'] then
              header.#{part}
            else
              default
            end
          end
        -
      end
    
    end


    tmail_attr :date, 'date', 'date'

    def date=( t )
      store 'Date', ''
      @header['date'].date = t
    end

    tmail_attr :to_addrs, 'to', 'addrs'
    tmail_attr :cc_addrs, 'cc', 'addrs'
    tmail_attr :bcc_addrs, 'bcc', 'addrs'

    def to( dflt = '' )
      arr = to_addrs
      if arr and arr[0] then
        arr[0]
      else
        dflt
      end
    end

    def to=( addr )
      store 'To', addr
    end

    def to_addrs=( arr )
      store 'To', ''
      @header['to'].addrs.replace arr
    end


    tmail_attr :from_addrs, 'from', 'addrs'

    def from_addrs=( arr )
      store 'From', ''
      @header['from'].addrs.replace arr
    end

    def from_addr( default = nil )
      if arr = from_addrs(nil) then
        arr[0] || default
      else
        default
      end
    end

    def from_address( default = nil )
      if a = from_addr(nil) then
        a.address
      else
        default
      end
    end

    alias from from_address

    def from_address=( addr )
      store 'From', addr
    end

    alias from= from_address=

    def from_phrase( default = nil )
      if a = from_addr(nil) then
        a.phrase
      else
        default
      end
    end


    tmail_attr :reply_to_addrs, 'reply-to', 'addrs'


    tmail_attr :sender, 'sender', 'addr'

    def sender=( addr )
      store 'Sender', addr
    end


    tmail_attr :subject, 'subject', 'body'

    def subject=( str )
      unless str then
        delete 'subject'
      else
        store 'Subject', str
      end
      str
    end


    tmail_attr :msgid, 'message-id', 'msgid'

    def msgid=( str )
      store 'Message-ID', str
      str
    end


    tmail_attr :references, 'references', 'msgids'

    def references=( arr )
      store 'References', ''
      @header['references'].refs.replace arr
    end


    tmail_attr :in_reply_to, 'in-reply-to', 'msgids'

    def in_reply_to=( arr )
      store 'In-Reply-To', ''
      @header['in-reply-to'].refs.replace arr
    end


    tmail_attr :mime_version, 'mime-version', 'version'

    def mime_version=( m, opt = nil )
      if opt then
        if head = self['Mime-Version'] then
          head.major = m
          head.minor = opt
        else
          store 'Mime-Version', "#{m}.#{opt}"
        end
      else
        store 'Mime-Version', m
      end
      m
    end


    tmail_attr :main_type, 'content-type', 'main'
    tmail_attr :sub_type, 'content-type', 'sub'

    def set_content_type( str, sub = nil, param = nil )
      if sub then
        main, sub = str, sub
      else
        main, sub = str.split('/', 2)
      end
      if head = self['content-type'] then
        head.main = main
        head.sub  = sub
      else
        store 'Content-Type', "#{main}/#{sub}"
      end
      self['content-type'].params.replace param if param

      str
    end

    alias content_type= set_content_type

    tmail_attr :charset, 'content-type', "params['charset']"

    def charset=( str )
      if head = self[ 'content-type' ] then
        head.params['charset'] = str
      else
        store "text/plain; charset=#{str}"
      end
      str
    end


    tmail_attr :encoding, 'content-transfer-encoding', 'encoding'

    def encoding=( str )
      store 'Content-Transfer-Encoding', str
      str
    end


    tmail_attr :disposition, 'content-disposition', 'disposition'

    def disposition=( str )
      store 'Content-Disposition', str
      str
    end

    def set_content_disposition( str, params = nil )
      if f = @header['content-disposition'] then
        f.disposition = str
      else
        f = store( 'Content-Disposition', str )
      end
      f.params.replace params if params
    end


    ###
    ### utils
    ###

    def destinations
      ret = []
      %w( to cc bcc ).each do |nm|
        if head = @header[nm] then
          head.addrs.each {|i| ret.push i.address }
        end
      end
      ret
    end

    def each_destination( &block )
      destinations.each do |i|
        if Address === i then
          yield i
        else
          i.each( &block )
        end
      end
    end

    alias each_dest each_destination


    def reply_addresses
      reply_to_addrs(nil) or from_addrs(nil)
    end

    def error_reply_addresses
      if s = sender(nil) then
        [s]
      else
        from_addrs(nil)
      end
    end

    def multipart?
      main_type('').downcase == 'multipart'
    end

  end   # class Mail

end   # module TMail
